use async_std::prelude::*;
use async_chat::utils::{self, ChatResult};
use async_std::io;
use async_std::net;

async fn send_commands(mut to_server: net::TcpStream) -> ChatResult<()> {
    println!("Polecenia:\n\
              join GRUPA\n\
              post GRUPA KOMUNIKAT...\n\
              Wpisz Control-D (w Uniksie) lub Control-Z (w Windowsieach) \
              by zamknąć połączenie.");

    let mut command_lines = io::BufReader::new(io::stdin()).lines();
    while let Some(command_result) = command_lines.next().await {
        let command = command_result?;
        // Definicję `parse_command` można znaleźć w repozytorium na GitHubie
        let request = match parse_command(&command) {
            Some(request) => request,
            None => continue,
        };

        utils::send_as_json(&mut to_server, &request).await?;
        to_server.flush().await?;
    }

    Ok(())
}

use async_chat::FromServer;

async fn handle_replies(from_server: net::TcpStream) -> ChatResult<()> {
    let buffered = io::BufReader::new(from_server);
    let mut reply_stream = utils::receive_as_json(buffered);

    while let Some(reply) = reply_stream.next().await {
        match reply? {
            FromServer::Message { group_name, message } => {
                println!("komunikat w grupie {}: {}", group_name, message);
            }
            FromServer::Error(message) => {
                println!("błąd komunikacji z serwerem: {}", message);
            }
        }
    }

    Ok(())
}

use async_std::task;

fn main() -> ChatResult<()> {
    let address = std::env::args().nth(1)
        .expect("Sposób użycia: client ADRES:PORT");

    task::block_on(async {
        let socket = net::TcpStream::connect(address).await?;
        socket.set_nodelay(true)?;

        let to_server = send_commands(socket.clone());
        let from_server = handle_replies(socket);

        from_server.race(to_server).await?;

        Ok(())
    })
}

use async_chat::FromClient;
use std::sync::Arc;

/// Parsuje wiersz (przesłany zapewne ze standardowego wejścia) jako `Request`
fn parse_command(line: &str) -> Option<FromClient> {
    let (command, rest) = get_next_token(line)?;
    if command == "post" {
        let (group, rest) = get_next_token(rest)?;
        let message = rest.trim_start().to_string();
        return Some(FromClient::Post {
            group_name: Arc::new(group.to_string()),
            message: Arc::new(message),
        });
    } else if command == "join" {
        let (group, rest) = get_next_token(rest)?;
        if !rest.trim_start().is_empty() {
            return None;
        }
        return Some(FromClient::Join {
            group_name: Arc::new(group.to_string()),
        });
    } else {
        eprintln!("Nieznane polecenie: {:?}", line);
        return None;
    }
}

/// Na podstawie łańcucha `input`, zwraca `Some((token, rest))`, gdzie `token` 
/// jest pierwszą sekwencją znaków (za wyjątkiem tak zwanych "białych znaków") 
/// w łańcuchu `input`, a `rest` jest pozostałą częścią łańcucha wejściowego.
/// Jeśli łańcuch nie zawiera żadnych odpowiednich znaków, funkcja zwraca `None`
fn get_next_token(mut input: &str) -> Option<(&str, &str)> {
    input = input.trim_start();

    if input.is_empty() {
        return None;
    }

    match input.find(char::is_whitespace) {
        Some(space) => Some((&input[0..space], &input[space..])),
        None => Some((input, "")),
    }
}
