//! Reading index files linearly from disk, a capability needed for merging
//! index files.

use std::fs::{self, File};
use std::io::prelude::*;
use std::io::{self, BufReader, SeekFrom};
use std::path::Path;
use byteorder::{LittleEndian, ReadBytesExt};
use crate::write::IndexFileWriter;

/// `IndexFileReader` liniowo odczytuje zawartość pliku indeksu od jego 
/// początku do końca. Oczywiście, nie trzeba wspominać, że prawdziwe indeksy
/// nie są używane w ten sposób. Ten sposób działania jest stosowany wyłącznie
/// do celu scalania kilku plików indeksów
///
/// Jedynym sposobme poruszania się po pliku do przodu jest użycie metody 
/// `.move_entry_to()`
pub struct IndexFileReader {
    /// Obiekt typu reader odczytujący faktyczne dane indeksu
    ///
    /// Używamy dwóch obiektów typu reader. Większość zawartości pliku stanowią
    /// dane indeksu. W pliku znajduje się także spis treści, zapisany osobno na
    /// końcu. Musimy odczytywać obie te części pliku, więc otwieramy go dwa razy
    main: BufReader<File>,

    /// Obiekt typu reader odczytujący spis treści. (Ponieważ spis jest zapisany
    /// na końcu pliku, musimy zacząć od przejścia w określone miejsce pliku 
    /// przy użyciu `seek`; patrz kod w `IndexFileReader::open_and_delete`.)
    contents: BufReader<File>,

    /// Kolejny wpis w spisie treści, jeśli jaki jest, lub `None` jeśli znaleźliśmy
    /// sie na końcu spis. `IndexFileReader` zawsze odczytuje z wyprzedzeniem o jeden
    /// spis i zapisuje go w tym miejscu
    next: Option<Entry>
}

/// Wpis spisu treści pliku indeksu.
///
/// Każdy wpis spisu treści musi być mały. Zawiera łańcuch znaków, `term`, podsumowanie
/// informacji na temat tego terminu (`df`) oraz wskaźnik do pełnych danych 
/// (`offset` and `nbytes`).
pub struct Entry {
    /// Termin jest słowem występującym w korpusie jednego lub kilku dokumentów.
    /// Plik indeksu zawiera informacje o dokumentach, w których to słowo 
    /// występuje
    pub term: String,

    /// Całkowita liczba dokumentów w korpusie zawierających ten termin
    pub df: u32,

    /// Przesunięcie danych indeksu dla danego terminu od początku pliku, w bajtach
    pub offset: u64,

    /// Długość danych indeksu dla danego terminu, liczona w bajtach
    pub nbytes: u64
}

impl IndexFileReader {
    /// Otwiera plik indeksu w celu odczytania jego zawartości od początku do końca.
    ///
    /// Operacja usuwa plik, co może nie działać prawidłowo w systemie Windows.
    /// Poprawki mile widziane! W systemie Unix działa to następująco: plik
    /// natychmiast znika z katalogu, lecz wciąż zajmuje miejsce na dysku, aż do 
    /// momentu zamknięcia pliku, co normalnie następuje po usunięciu zmiennej
    /// `IndexFileReader` 
    pub fn open_and_delete<P: AsRef<Path>>(filename: P) -> io::Result<IndexFileReader> {
        let filename = filename.as_ref();
        let mut main_raw = File::open(filename)?;

        // Wczytanie nagłówka pliku
        let contents_offset = main_raw.read_u64::<LittleEndian>()?;
        println!("otworzono plik {}, początek spisu treści w lokalizacji {}", filename.display(), contents_offset);

        // Ponowne otworzenie pliku, tak że dysponujemy dwoma punktami początkowymi
        // odczytu. Przesunięcie punktu odczytu treści do odpowiedniego miejsca 
        // pliku. Konfiguracja buforowania
        let mut contents_raw = File::open(filename)?;
        contents_raw.seek(SeekFrom::Start(contents_offset))?;
        let main = BufReader::new(main_raw);
        let mut contents = BufReader::new(contents_raw);

        // Zawsze odczytujemy z wyprzedzeniem o jeden wpis, więc tu wczytujemy pierwszy
        let first = IndexFileReader::read_entry(&mut contents)?;

        fs::remove_file(filename)?;  // YOLO

        Ok(IndexFileReader {
            main: main,
            contents: contents,
            next: first
        })
    }

    /// Odczytuje następny wpis spisu treści.
    ///
    /// Zwraca `Ok(None)` jeśli dotarliśmy na koniec pliku
    fn read_entry(f: &mut BufReader<File>) -> io::Result<Option<Entry>> {
        // Jeśli pierwszy odczyt zakończy się zwróceniem `UnexpectedEof`
        // traktujemy to jako sukces, choć żaden wpis nie został wczytany
        let offset = match f.read_u64::<LittleEndian>() {
            Ok(value) => value,
            Err(err) =>
                if err.kind() == io::ErrorKind::UnexpectedEof {
                    return Ok(None)
                } else {
                    return Err(err)
                }
        };

        let nbytes = f.read_u64::<LittleEndian>()?;
        let df = f.read_u32::<LittleEndian>()?;
        let term_len = f.read_u32::<LittleEndian>()? as usize;
        let mut bytes = Vec::with_capacity(term_len);
        bytes.resize(term_len, 0);
        f.read_exact(&mut bytes)?;
        let term = match String::from_utf8(bytes) {
            Ok(s) => s,
            Err(_) => return Err(io::Error::new(io::ErrorKind::Other, "błąd unicode"))
        };

        Ok(Some(Entry {
            term: term,
            df: df,
            offset: offset,
            nbytes: nbytes
        }))
    }

    /// Pożycza referencję do następnego wpisu w spisie treści.
    /// (Ponieważ zawsze odczytujemy z wyprzedzeniem o jeden wpis, wywołanie tej
    /// metody nie może zakończyć się niepowodzeniem.)
    ///
    /// Zwraca `None` po dotarciu do końca pliku
    pub fn peek(&self) -> Option<&Entry> { self.next.as_ref() }

    /// True jeśli jeśli następny wpis dotyczy podanego terminu
    pub fn is_at(&self, term: &str) -> bool {
        match self.next {
            Some(ref e) => e.term == term,
            None => false
        }
    }

    /// Kopiuje bieżący wpis do określonego strumienia wyjściowego, a następnie
    /// odczytuje nagłówek następnego wpius
    pub fn move_entry_to(&mut self, out: &mut IndexFileWriter) -> io::Result<()> {
        // Ten blok ogranicza zakres porzyczania `self.next` (dla `e`),
        // gdyż po zakończeniu tego bloku będziemy chcieli wykonać przypisanie 
        // do `self.next`
        {
            let e = self.next.as_ref().expect("brak wpisu do przeniesienia");
            if e.nbytes > usize::max_value() as u64 {
                // This can only happen on 32-bit platforms.
                return Err(io::Error::new(io::ErrorKind::Other,
                                          "wpis do indeksu jest zbyt duży by zapisać go na komputrze"));
            }
            let mut buf = Vec::with_capacity(e.nbytes as usize);
            buf.resize(e.nbytes as usize, 0);
            self.main.read_exact(&mut buf)?;
            out.write_main(&buf)?;
        }

        self.next = Self::read_entry(&mut self.contents)?;
        Ok(())
    }
}
