/** Klasa <code>Angle</code> jest klas opakowujc. Zawiera jedno prywatne pole typu <code>double</code>,
  * przechowujce miar konta wyraonego w radianach.
  */
public class Angle {
/** Przechowuje miar kta reprezentowanego przez obiekt, podan w radianach. 
  */	
	private double x;
	
/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt <code>x rad</code>.
  * @param x liczba typu <code>double</code>, miara kta w radianach.
  */		
	public Angle(double x) {
	    this.x = x;
    }
    
/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie (cakowitej) stopni z przedziau &lt;0, 360&gt;.
  * @param deg miara kta w stopniach, liczba cakowita typu 
  * <code>int</code> z przedziau &lt;0, 360&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;.
  */    
    public Angle(int deg) {
        if (deg < 0 || deg > 360)
            throw new IllegalArgumentException("Liczba stopni poza zakresem.");
        x = deg/180.0*Math.PI;
    }
/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie stopni i minut.
  * @param deg liczba stopni, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 360&gt;,
  * @param min liczba minut, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;,
  * @throws IllegalArgumentException gdy liczba minut nie mieci si 
  * w przedziale &lt;0, 60&gt;.
  */
    public Angle(int deg, int min) {
        if (deg < 0 || deg > 360)
            throw new IllegalArgumentException("Liczba stopni poza zakresem.");
        if (min < 0 || min > 60)
            throw new IllegalArgumentException("Liczba minut poza zakresem.");
        x = (deg+min/60.0)/180.0*Math.PI;
    }
    
/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie stopni, minut i sekund.
  * @param deg liczba stopni, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 360&gt;,
  * @param min liczba minut, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;,
  * @param sek liczba sekund, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;,
  * @throws IllegalArgumentException gdy liczba minut nie mieci si 
  * w przedziale &lt;0, 60&gt;,
  * @throws IllegalArgumentException gdy liczba sekund nie mieci si 
  * w przedziale &lt;0, 60&gt;.
  */ 
    public Angle(int deg, int min, int sek) {
        if (deg < 0 || deg > 360)
            throw new IllegalArgumentException("Liczba stopni poza zakresem.");
        if (min < 0 || min > 60)
            throw new IllegalArgumentException("Liczba minut poza zakresem.");
        if (sek < 0 || sek > 60)
            throw new IllegalArgumentException("Liczba sekund poza zakresem.");
        x = (deg+min/60.0+sek/3600.0)/180.0*Math.PI;
    }

/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * w postaci acucha znakw (2230'15") liczbie stopni, minut i sekund.
  * @param st acuch znakw postaci <code>xxxyy'zz"</code>, gdzie 
  * <code>x</code>x, <code>y</code> i <code>z</code> oznaczaj 
  * odpowiednio liczb stopni, minut i sekund. Liczba minut i sekund 
  * nale do przedziau &lt;0, 59&gt;,
  * @throws IllegalArgumentException gdy wejciowy acuch znakw nie 
  * odpowiada poprawnej mierze kta.
  * <p><b>Poprawne dane:</b> <code>2345'30"</code>, <code>2345'</code>, 
  * <code>23</code>, <code>045'30"</code> <code>-1525'</code>.</p> 
  * <p><b>Bdne dane:</b> <code>12-05'</code>, <code>2o30'30"</code>, 
  * <code>2275</code>.</p>
  */   
    public Angle(String st) {
        int deg, min = 0, sek = 0;
        int p = st.indexOf("\u00B0"); // stopie
        int q = st.indexOf("\'");     // minuta ktowa
        int r = st.indexOf("\"");     // sekunda ktowa
        try {
            deg = Integer.parseInt(st.substring(0, p));
            if (q > 0) {
                min = Integer.parseInt(st.substring(p+1, q));
                if (r > 0) 
                   sek = Integer.parseInt(st.substring(q+1, r));
            }
        } catch (Exception e) {
	        throw new IllegalArgumentException(st);
        }
        if ((min < 0) || (min > 59) || (sek < 0) || (sek > 59))
            throw new IllegalArgumentException(st);
        boolean znak =false;
        if (deg < 0) {
            znak = true;
            deg = -deg;
        }
        this.x = (deg+min/60.0+sek/3600.0)/180.0*Math.PI;
        if (znak)
            this.x = -this.x;
    }
/** Tworzy nowy obiekt <code>Angle</code> reprezentujcy kt nachylenia 
  * prostej wyznaczonej przez punkt <code>P(x, y)</code> i rodek ukadu 
  * wsprzdnych <code>(0, 0)</code> do osi <code>OX</code>.
  * @param x wsprzdna <code>x</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>. 
  * @param y wsprzdna <code>y</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>.
  * @throws IllegalArgumentException gdy <code>x = 0</code> 
  * i <code>y = 0</code>.
  */
    public Angle(double x, double y) {
        if (x == 0 && y == 0)
            throw new IllegalArgumentException();
        if (y >= 0)
            this.x = Math.atan2(y, x);
        else
            this.x = 2*Math.PI+Math.atan2(y, x);
    }
        
/** Zwraca warto funkcji sinus kta reprezentowanego przez obiekt.
  */    
	public double sin() { /* sinus */
		return Math.sin(x);
    }
    
/** Zwraca warto funkcji cosinus kta reprezentowanego przez obiekt.
  */     
    public double cos() { /* cosinus */
		return Math.cos(x);
    }
    
/** Zwraca warto funkcji tangens kta reprezentowanego przez obiekt.
  */     
    public double tan() { /* tangens */
		return Math.tan(x);
    }
    
/** Zwraca warto funkcji cotangens kta reprezentowanego przez obiekt.
  */     
    public double cot() { /* cotangens */
		return 1/Math.tan(x);
    }
    
/** Zwraca warto funkcji secans kta reprezentowanego przez obiekt.
  */ 
    public double sec() { /* secans */
		return 1/Math.cos(x);
    }
    
/** Zwraca warto funkcji cosecans kta reprezentowanego przez obiekt.
  */ 
    public double csc() { /* cosecans */
		return 1/Math.sin(x);
    }
/** Zwraca miar kta reprezentowanego przez obiekt wyraon w radianach.
  */    
    public double radian() {
	    return x;
    }
    
/** Zwraca miar kta reprezentowanego przez obiekt wyraon w stopniach.
  */    
    public double degree() {
	    return x/Math.PI*180;
    }
    
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto sinusa kta jest rwna podnej wartoci parametru 
  * <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>&lt; -&pi;/2, &pi;/2&gt;</code>.
  * @param x liczba typu <code>double</code> naleca do przedziau 
  * <code>&lt;-1, 1&gt;</code>.
  * @throws IllegalArgumentException gdy <code>x</code> nie naley do 
  * przedziau <code>&lt;-1, 1&gt;</code>.
  */
    public void setOfSin(double x) {
	    if (x >= -1 && x <= 1)
	        this.x = Math.asin(x);
	    else
	        throw new IllegalArgumentException();
    }
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto cosinusa kta jest rwna podnej wartoci parametru 
  * <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>&lt;0, &pi;&gt;</code>.
  * @param x liczba typu <code>double</code> naleca do przedziau 
  * <code>&lt;-1, 1&gt;</code>.
  * @throws IllegalArgumentException gdy <code>x</code> nie naley do 
  * przedziau <code>&lt;-1, 1&gt;</code>.
  */    
    public void setOfCos(double x) {
	    if (x >= -1 && x <= 1)
	        this.x = Math.acos(x);
	    else
	        throw new IllegalArgumentException();
    }
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto tangensa kta jest rwna podnej wartoci parametru 
  * <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>(-&pi;/2, &pi;/2)</code>.
  * @param x liczba typu <code>double</code>.
  */    
    public void setOfTan(double x) {
	    this.x = Math.atan(x);
    }
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto cotangensa kta jest rwna podnej wartoci 
  * parametru <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>(0, &pi;)</code>.
  * @param x liczba typu <code>double</code>.
  */   
    public void setOfCot(double x) {
	    if (x == 0.0)
	        this.x = Math.PI/2;
	    else
	        this.x = Math.PI+Math.atan(1/x);
    }
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto secansa kta jest rwna podnej wartoci parametru 
  * <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>&lt;0, &pi;&gt;</code>.
  * @param x liczba typu <code>double</code> naleca do zbioru 
  * <code>&lt;-&#8734;, -1&gt; &#8746; &lt;1, +&#8734;&gt;</code>.
  * @throws IllegalArgumentException gdy <code>x</code> naley do 
  * przedziau <code>(-1, 1)</code>.
  */      
    public void setOfSec(double x) {
	    if (x > -1 && x < 1)
	        throw new IllegalArgumentException();
	    else    
	        this.x = Math.acos(1/x);
    }
    
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar, 
  * dla ktrej warto cosecansa kta jest rwna podnej wartoci parametru 
  * <code>x</code>. Otrzymana miara kta naley do przedziau 
  * <code>&lt;-&pi;/2, &pi;/2&gt;</code>.
  * @param x liczba typu <code>double</code> naleca do zbioru 
  * <code>&lt;-&#8734;, -1&gt; &#8746; &lt;1, +&#8734;&gt;</code>.
  * @throws IllegalArgumentException gdy <code>x</code> naley do 
  * przedziau <code>(-1, 1)</code>.
  */    
    public void setOfCsc(double x) {
	    if (x > -1 && x < 1)
	        throw new IllegalArgumentException();
	    else    
	        this.x = Math.asin(1/x);
    }
/** Zmienia warto miary kta reprezentowanego przez obiekt na miar 
  * kta nachylenia prostej wyznaczonej przez punkt <code>P(x, y)</code> 
  * i rodek ukadu wsprzdnych <code>(0, 0)</code> do osi 
  * <code>OX</code> ukadu wsprzdnych.
  * @param x wsprzdna <code>x</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>. 
  * @param y wsprzdna <code>y</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>.
  * @throws IllegalArgumentException gdy <code>x = 0</code> 
  * i <code>y = 0</code>.
  */
    public void setOfPoint(double x, double y) {
	    if (x == 0 && y == 0)
	        throw new IllegalArgumentException();
        if (y >= 0)
            this.x = Math.atan2(y, x);
        else
            this.x = 2*Math.PI+Math.atan2(y, x);
    }

/** Zwraca obiekt <code>Angle</code> reprezentujcy kt <code>x rad</code>.
  * @param x liczba typu <code>double</code>, miara kta w radianach.
  */    
    public static Angle valueOf(double x) {
	    return new Angle(x);
    }
/** Zwraca obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie (cakowitej) stopni z przedziau &lt;0, 360&gt;.
  * @param deg miara kta w stopniach, liczba cakowita typu 
  * <code>int</code> z przedziau &lt;0, 360&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;.
  */      	
    public static Angle valueOf(int deg) {
	    return new Angle(deg);
    }

/** Zwraca obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie stopni i minut.
  * @param deg liczba stopni, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 360&gt;,
  * @param min liczba minut, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;,
  * @throws IllegalArgumentException gdy liczba minut nie mieci si 
  * w przedziale &lt;0, 60&gt;.
  */    
    public static Angle valueOf(int deg, int min) {
	    return new Angle(deg, min);
    }
    
/** Zwraca obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * liczbie stopni, minut i sekund.
  * @param deg liczba stopni, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 360&gt;,
  * @param min liczba minut, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;,
  * @param sek liczba sekund, liczba cakowita typu <code>int</code> 
  * z przedziau &lt;0, 60&gt;.
  * @throws IllegalArgumentException gdy liczba stopni nie mieci si 
  * w przedziale &lt;0, 360&gt;,
  * @throws IllegalArgumentException gdy liczba minut nie mieci si 
  * w przedziale &lt;0, 60&gt;,
  * @throws IllegalArgumentException gdy liczba sekund nie mieci si 
  * w przedziale &lt;0, 60&gt;.
  */    
    public static Angle valueOf(int deg, int min, int sek) {
	    return new Angle(deg, min, sek);
    }
    
/** Zwraca obiekt <code>Angle</code> reprezentujcy kt o podanej 
  * w postaci acucha znakw (2230'15") liczbie stopni, minut i sekund.
  * @param st acuch znakw postaci <code>xxxyy'zz"</code>, gdzie 
  * <code>x</code>x, <code>y</code> i <code>z</code> oznaczaj 
  * odpowiednio liczb stopni, minut i sekund. Liczba minut i sekund 
  * nale do przedziau &lt;0, 59&gt;,
  * @throws IllegalArgumentException gdy wejciowy acuch znakw nie 
  * odpowiada poprawnej mierze kta.
  * @throws NumberFormatException gdy w wejciowym acuchu znajduj si 
  * niepoprawna posta liczby.
  * <p><b>Poprawne dane:</b> <code>2345'30"</code>, <code>2345'</code>, 
  * <code>23</code>, <code>45'30"</code>, <code>45'</code>, <code>35"</code>,
  * <code>115"</code> <code>-1525'</code>.</p> 
  * <p><b>Bdne dane:</b> <code>12-05'</code>, <code>2o30'30"</code>, 
  * <code>2275</code>, <code>25</code>.</p>
  */    
    public static Angle valueOf(String st) {
        int deg = 0, min = 0, sek = 0;
        int pst = st.indexOf("\u00B0");      // stopie
        if (pst == -1) {                     // nie ma symbolu stopnia
            int pmin = st.indexOf("\'");     // minuta ktowa
            if (pmin == -1) {                // nie ma symbolu minuty
                int psek = st.indexOf("\""); // sekunda ktowa
                if (psek == -1) {            // nie ma symbolu sekundy
                    throw new IllegalArgumentException("Brak jednostek.");
                } else                       // tylko sekundy
                   sek = Integer.parseInt(st.substring(pmin+1, psek));
            } else {
                min = Integer.parseInt(st.substring(0, pmin));
                int psek = st.indexOf("\""); // sekunda ktowa
                if (psek != -1)
                    sek = Integer.parseInt(st.substring(pmin+1, psek));
            }
        } else { // jest symbol stopnia
            deg = Integer.parseInt(st.substring(0, pst));
            int pmin = st.indexOf("\'");     // minuta ktowa
            if (pmin == -1) {
                int psek = st.indexOf("\""); // sekunda ktowa
                if (psek != -1) 
                    sek = Integer.parseInt(st.substring(pst+1, psek)); 
                } else {
                    min = Integer.parseInt(st.substring(pst+1, pmin));
                    int psek = st.indexOf("\""); // sekunda ktowa
                    if (psek != -1)	
                        sek = Integer.parseInt(st.substring(pmin+1, psek));
                }
        }
        boolean minus = false;
        if (deg < 0) {
	        deg = -deg;
	        minus = true;
        }
        Angle tmp = new Angle(deg, min, sek);
        if (minus)
            tmp.x = -tmp.x;
        return tmp;
    }
    
/** Zwraca obiekt <code>Angle</code> reprezentujcy kt nachylenia 
  * prostej wyznaczonej przez punkt <code>P(x, y)</code> i rodek ukadu 
  * wsprzdnych <code>(0, 0)</code> do osi <code>OX</code>.
  * @param x wsprzdna <code>x</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>. 
  * @param y wsprzdna <code>y</code> punktu <code>P(x, y)</code>, 
  * liczba typu <code>double</code>.
  * @throws IllegalArgumentException gdy <code>x = 0</code> 
  * i <code>y = 0</code>.
  */    
    public static Angle valueOf(double x, double y) {
        return new Angle(x, y);
    }
    
/** Zwraca acuch znakw (postaci 4530'15") przedstawiajcy miar kta 
  * reprezentowanego przez obiekt, wyraon w stopniach, minutach 
  * i sekundach.
  */
    @Override public String toString() {
        if (this.x == 0) 
            return "0\u00B0";
        double x = this.x/Math.PI*180;
        StringBuffer str = new StringBuffer();
        if (x < 0) {
            str.append("-");
            x = -x;
        }
        int deg = (int)Math.floor(x);
        x = (x-Math.floor(x))*60;
        int min = (int)Math.floor(x);
        x = (x-Math.floor(x))*60;
        int sek = (int)Math.floor(x+0.5);
        if (sek == 60) {
            sek = 0;
            ++min;
        }
        if (min == 60) {
            min = 0;
            ++deg;
        }
        if (deg != 0) {
            str.append(deg);
            str.append("\u00B0");
        }
        if (min != 0 || (deg != 0 && sek != 0)) 
            str.append(min).append("\'");
        if (sek != 0) 
            str.append(sek).append("\"");
        return str.toString();
    }
		
    public Angle add(Angle x) {
	    return new Angle(this.x+x.x);
    }
    public Angle sub(Angle x) {
	    return new Angle(this.x-x.x);
    }
    public static Angle sum(Angle x, Angle y) {
	    return x.add(y);
    }
    public static Angle diff(Angle x, Angle y) {
	    return x.sub(y);
    }
    public Angle mult(double a) {
	    return new Angle(a*this.x);
    }
    public Angle mult(int n) {
	    return new Angle(n*this.x);
    }
    public Angle div(double a) {
	    if (a == 0.0)
	        throw new ArithmeticException("Dzielenie przez 0!");
	    return new Angle(a*this.x/a);
    }
    public Angle div(int n) {
	    if (n == 0)
	        throw new ArithmeticException("Dzielenie przez 0!");
	    return new Angle(this.x/n);
    }
    public static Angle prod(Angle x, double y) {
	    return x.mult(y);
    }
    public static Angle prod(Angle x, int n) {
	    return x.mult(n);
    }
    public static Angle quot(Angle x, double y) {
	    return x.div(y);
    }
    public static Angle quot(Angle x, int n) {
	    return x.div(n);
    }
    public static Angle compl(Angle x) {
	    if (x.x < 0.0 && x.x > RIGHT_ANGLE.x)
	        throw new IllegalArgumentException();
	    return RIGHT_ANGLE.sub(x);
    }
    public static Angle suppl(Angle x) {
	    if (x.x < 0.0 && x.x > STRAIGHT_ANGLE.x)
	        throw new IllegalArgumentException();
	    return STRAIGHT_ANGLE.sub(x);
    }
    public static final Angle RIGHT_ANGLE = new Angle(Math.PI/2);
    public static final Angle STRAIGHT_ANGLE = new Angle(Math.PI);
    public static final Angle FULL_ANGLE = new Angle(2*Math.PI);
    public static final Angle RADIAN = new Angle(1.0);
    public static final Angle DEGREE = new Angle(1);
    public static final Angle ARCMINUTE = new Angle(0, 1);
    public static final Angle ARCSECOND = new Angle(0, 0, 1);
}
