package main

import (
  "net/http"

  "gopkg.in/mgo.v2"
  "gopkg.in/mgo.v2/bson"
)

type poll struct {
  ID      bson.ObjectId   `bson:"_id" json:"id"`
  Title   string          `json:"title" bson:"title"`
  Options []string        `json:"options"`
  Results map[string]int  `json:"results,omitempty"`
  APIKey  string          `json:"apikey"`
}

func (s *Server) handlePolls(w http.ResponseWriter, r *http.Request) {
  switch r.Method {
  case "GET":
    s.handlePollsGet(w, r)
    return
  case "POST":
    s.handlePollsPost(w, r)
    return
  case "DELETE":
    s.handlePollsDelete(w, r)
    return
  case "OPTIONS":
    w.Header().Add("Access-Control-Allow-Methods", "DELETE")
    respond(w, r, http.StatusOK, nil)
    return    
  }
  // nie znaleziono
  respondHTTPErr(w, r, http.StatusNotFound)
}

func (s *Server) handlePollsGet(w http.ResponseWriter, r *http.Request) {
  session := s.db.Copy()
  defer session.Close()
  c := session.DB("ballots").C("polls")
  var q *mgo.Query
  p := NewPath(r.URL.Path)
  if p.HasID() {
    // pobranie konkretnej ankiety
    q = c.FindId(bson.ObjectIdHex(p.ID))
  } else {
    // pobranie wszystkich ankiet
    q = c.Find(nil)
  }
  var result []*poll
  if err := q.All(&result); err != nil {
    respondErr(w, r, http.StatusInternalServerError, err)
    return
  }
  respond(w, r, http.StatusOK, &result)
}

func (s *Server) handlePollsPost(w http.ResponseWriter, r *http.Request) {
  session := s.db.Copy()
  defer session.Close()
  c := session.DB("ballots").C("polls")
  var p poll
  if err := decodeBody(r, &p); err != nil {
    respondErr(w, r, http.StatusBadRequest,  "Nie udało się odczytać danych ankiety z żądania", err)
    return
  }
  apikey, ok := APIKey(r.Context())
  if ok {
    p.APIKey = apikey
  }
  p.ID = bson.NewObjectId()
  if err := c.Insert(p); err != nil {
    respondErr(w, r, http.StatusInternalServerError, "Nie udało się zapisać ankiety w bazie danych", err)
    return
  }
  w.Header().Set("Location", "polls/"+p.ID.Hex())
  respond(w, r, http.StatusCreated, nil)
}

func (s *Server) handlePollsDelete(w http.ResponseWriter, r *http.Request) {
  session := s.db.Copy()
  defer session.Close()
  c := session.DB("ballots").C("polls")
  p := NewPath(r.URL.Path)
  if !p.HasID() {
    respondErr(w, r, http.StatusMethodNotAllowed, "Nie można usunąć wszystkich ankiet")
    return
  }
  if err := c.RemoveId(bson.ObjectIdHex(p.ID)); err != nil {
    respondErr(w, r, http.StatusInternalServerError, "Nie udało się usunąć ankiety", err)
    return
  }
  respond(w, r, http.StatusOK, nil) // w porządku
}
