#ifndef SERVICE_H
#define SERVICE_H

// Biblioteka standardowa
#include <iostream>
#include <functional>

// Boost ASIO
#include <boost/asio.hpp>

// Nie jest dobrym pomysem umieszczenie deklaracji using w pliku nagwkowym.
// ale w przypadku tego niewielkiego przykadu jako to przeyjemy.
using boost::asio::ip::tcp;


namespace detail {

/**
 * Struktura zawierajca warto dowolnego typu
 * oraz informacje kontekstowe - gniazdo dla klienta.
 *
 * Przekazywanie surowego wskanik do gniazda jest niebezpieczne.
 * Zazwyczaj zamiast niego przekazywalibymy identyfikator klienta,
 * a centralny komponent serwera zarzdzaby wszystkimi klientami.
 */
template <typename MessageType>
struct with_client {
    MessageType value;
    tcp::socket *socket;

    void reply(const std::string& message) const
    {
        boost::asio::async_write(
                *socket,
                boost::asio::buffer(message, message.length()),
                                 [](auto, auto) {});
    }
};

} // szczegy przestrzeni nazw

/**
 * Funkcja, ktra tworzy instancj klasy with_client
 * majc dan warto i gniazdo.
 */
template <typename MessageType>
auto make_with_client(MessageType&& value, tcp::socket* socket)
{
    return ::detail::with_client<MessageType>{
        std::forward<MessageType>(value), socket};
}

/**
 * Podnosi dowoln funkcj przeksztacajc typ T1 na T2
 * do funkcji przeksztacajcej typ with_client<T1> na with_client<T2>.
 */
template <typename F>
auto lift_with_client(F&& function)
{
    return [function = std::forward<F>(function)] (auto &&ws) {
        return make_with_client(std::invoke(function, ws.value), ws.socket);
    };
}

/**
 * Konwertuje funkcj przeksztacajc typ T1 na T2 na funkcj
 * przeksztacajc typ with_client<T1> na T2.
 * Nie jest to przydatne dla wszystkich monad, ale potrzebujemy tej funkcji,
 * aby mc uy transformacji filter dla strumieni, ktre wysyaj wartoci ze wskanikiem do gniazda.
 */
template <typename F>
auto apply_with_client(F&& function)
{
    return [function = std::forward<F>(function)] (auto &&ws) {
        return std::invoke(function, std::forward<decltype(ws)>(ws).value);
    };
}




/**
 * Klasa obsugujca sesj.
 *
 * Odczytuje dane wysane przez klienta wiersz po wierszu,
 * a nastpnie wysya kady z wierszy w postaci oddzielnej wiadomoci.
 */
template <typename EmitFunction>
class session: public std::enable_shared_from_this<session<EmitFunction>> {
public:
    session(tcp::socket&& socket, EmitFunction emit)
        : m_socket(std::move(socket))
        , m_emit(emit)
    {
    }

    void start()
    {
        do_read();
    }

private:
    using shared_session = std::enable_shared_from_this<session<EmitFunction>>;

    void do_read()
    {
        // Uzyskiwanie wspdzielonego wskanika do tej instancji,
		// aby przechwyci go w wyraeniu lambda.
        auto self = shared_session::shared_from_this();
        boost::asio::async_read_until(
            m_socket, m_data, '\n',
            [this, self](const boost::system::error_code& error,
                         std::size_t size) {
                if (!error) {
                    // Odczytywanie wiersza tekstu
					// i przekazywanie go tym klientom, ktrzy nas nasuchuj.
                    std::istream is(&m_data);
                    std::string line;
                    std::getline(is, line);
                    m_emit(make_with_client(std::move(line), &m_socket));

                    // Zaplanowanie odczytywania kolejnego wiersza.
                    do_read();
                }
            });
    }

    tcp::socket m_socket;
    boost::asio::streambuf m_data;
    EmitFunction m_emit;
};

/**
 * Tworzy wspdzielony wskanik do sesji okrelanej przez gniazdo
 * oraz definiuje funkcj, ktra zostanie uyta przez sesj
 * do wysyania wiadomoci.
 */
template <typename Socket, typename EmitFunction>
auto make_shared_session(Socket&& socket, EmitFunction&& emit)
{
    return std::make_shared<session<EmitFunction>>(
            std::forward<Socket>(socket),
            std::forward<EmitFunction>(emit));
}

/**
 * Klasa service obsuguje poczenia od klientw
 * oraz wysya wiadomoci odebrane od nich.
 */
class service {
public:
    using value_type = ::detail::with_client<std::string>;

    explicit service(boost::asio::io_service& service,
                     unsigned short port = 42042);

    service(const service &other) = delete;
    service(service &&other) = default;

    template <typename EmitFunction>
    void on_message(EmitFunction emit)
    {
        m_emit = emit;
        do_accept();
    }

private:
    void do_accept();

    tcp::acceptor m_acceptor;
    tcp::socket m_socket;
    std::function<void(::detail::with_client<std::string>&&)> m_emit;

    friend std::ostream& operator<< (std::ostream& out, const service& service)
    {
        return out << "obiekt service";
    }
};

#endif
