// vim: zdefiniuj parametr set undofile:
// Program: 06-string-concatenation

#include <iostream>
#include <string>
#include <tuple>

// Użycie szablonów wyrażeń w celu łączenia łańcuchów

template <typename... Strings>
class lazy_string_concat_helper;

template <typename LastString, typename... Strings>
class lazy_string_concat_helper<LastString,
                                Strings...> {
private:
    // Przechowuje kopię oryginalnego łańcucha
    LastString data;

    // Przechowuje strukturę, która zawiera inne łańcuchy
    lazy_string_concat_helper<Strings...> tail;

public:
    lazy_string_concat_helper(
            LastString data,
            lazy_string_concat_helper<Strings...> tail)
        : data(data)
        , tail(tail)
    {
    }

    // Oblicza rozmiar wszystkich połączonych ze sobą łańcuchów
    int size() const
    {
        return data.size() + tail.size();
    }

    // Struktura przechowuje łańcuchy w odwrotnej kolejności: zmienna składowa data zawiera łańcuch,
	// który jest dostarczany jako ostatni, więc musi się znaleźć na końcu bufora
    template <typename It>
    void save(It end) const
    {
        const auto begin = end - data.size();
        std::copy(data.cbegin(), data.cend(),
                  begin);
        tail.save(begin);
    }

    // Aby zamienic definicję wyrażenia na prawdziwy łańcuch,
	// należy przydzielić wystarczającą ilość pamięci i rozpocząć kopiowanie łańcuchów do niej.
    operator std::string() const
    {
        std::string result(size(), '\0');
        save(result.end());
        return result;
    }

    // Tworzy nową instancję struktury z zawartym w niej pojedynczym łańcuchem
    lazy_string_concat_helper<std::string,
                              LastString,
                              Strings...>
    operator+(const std::string& other) const
    {
        return lazy_string_concat_helper
               <std::string, LastString, Strings...>(
                   other,
                   *this
               );
    }
};



template <>
class lazy_string_concat_helper<> {
public:
    lazy_string_concat_helper()
    {
    }

    int size() const
    {
        return 0;
    }

    template <typename It>
    void save(It) const
    {
    }

    lazy_string_concat_helper<std::string>
    operator+(const std::string& other) const
    {
        return lazy_string_concat_helper<std::string>(
                other,
                *this
            );
    }
};


lazy_string_concat_helper<> lazy_concat;

int main(int argc, char* argv[])
{
    std::string name = "Janina";
    std::string surname = "Nowak";

    // W przypadku typu std::string nie możesz przeciążyć operatora +, więc użyj małej sztuczki,
	// aby wymusić wykorzystanie naszej struktury przez dołączenie jej do instancji tego typu
    const std::string fullname =
        lazy_concat + name + " " + surname;

    std::cout << fullname << std::endl;
}

