
// Biblioteka standardowa
#include <iostream>
#include <string>

// Biblioteka JSON
#include <nlohmann/json.hpp>
using json = nlohmann::json;

// Biblioteka Catch
#define CATCH_CONFIG_MAIN
#include <catch2/catch.hpp>

// Narzdzia
#include "expected.h"
#include "mtry.h"
#include "trim.h"

// Implementacja usugi
#include "service.h"

/**
 * Dla funkcji, ktre zwracaj obiekty json. Gdy ich wykonanie
 * moe si nie powie, zwracaj wyjtek.
 */
using expected_json = expected<json, std::exception_ptr>;

/**
 * Podstawowe dane zakadki. Struktura przechowuje adres URL i nazw.
 */
struct bookmark_t {
    std::string url;
    std::string text;
};

/**
 * Tworzy acuch na podstawie zakadki w nastpujcym formacie:
 * [tekst](url)
 */
std::string to_string(const bookmark_t& page)
{
    return "[" + page.text + "](" + page.url + ")";
}

/**
 * Zapisuje informacje zwizane z zakadk w nastpujcym formacie:
 * [tekst](url)
 */
std::ostream& operator<<(std::ostream& out, const bookmark_t& page)
{
    return out << "[" << page.text << "](" << page.url << ")";
}

/**
 * Typ, ktry zawiera zakadk lub bd (wyjtek).
 */
using expected_bookmark = expected<bookmark_t, std::exception_ptr>;

/**
 * Prbuje odczyta zakadk z obiektu JSON. W przypadku bdu
 * nastpi zwrcenie wyjtku.
 */
expected_bookmark bookmark_from_json(const json& data)
{
    return mtry([&] {
            return bookmark_t { data.at("FirstURL"), data.at("Text") };
        });
}

TEST_CASE("Zliczanie znakw nowego wiersza w acuchu", "[program-logic]")
{
    using namespace ranges::v3;

    // Podnosimy funkcje transform i filter w taki sposb,
	// by mogy obsugiwa typ with_client<T>, ktry dodaje
	// informacj o gniedzie do danej wartoci, dziki czemu
	// moemy odpowiada klientom.
    auto transform = [](auto f) {
            return view::transform(lift_with_expected_reply(f));
        };
    auto filter = [](auto f) {
            return view::filter(apply_with_expected_reply(f));
        };

	// Koncepcja ujcia nie istnieje dla zakresw.
	// Moemy jednak po prostu uy algorytmu view::transform,
	// aby wymusi, by funkcja f zostaa wyznaczona dla
	//  wszystkich wartoci zakresu, ktry wysyamy do ujcia.
    auto sink = [](auto f) {
            return view::transform([f] (auto &&ws) {
                    f(ws);
                    return ws.expected_reply;
                });
        };


	// Cakiem przyjemna funkcja tworzca przypadki testowe
    auto test = [] (const std::string& message, const std::string& expected_reply) {
        return with_expected_reply<std::string>{message, expected_reply};
        };

    std::vector<with_expected_reply<std::string>> source {
        test("", ""),

        test("witaj",
             "BD: danie niezrozumiae.\n"),

        test("{}",
             "BD: danie niezrozumiae.\n"),

        test("{\"FirstURL\" : \"http://www.iso.org/\",\"Text\" : \"ISO\"}",
             "BD: odnonik nie dotyczy C++.\n"),

        test("{\"FirstURL\" : \"http://isocpp.org/\",\"Text\" : \"ISO C++ - Strona oficjalna\"}",
             "OK: [ISO C++ - Strona oficjalna](http://isocpp.org/)\n")
    };

    auto pipeline =
        source
            | transform(trim)

            // Ignorujemy komentarze i puste wiadomoci.
            | filter([] (const std::string &message) {
                return message.length() > 0 && message[0] != '#';
            })

            // Prbujemy parsowa dane wejciowe.
            | transform([] (const std::string &message) {
                return mtry([&] {
                    return json::parse(message);
                });
            })

            // Przeksztacamy wynik na zakadk.
            | transform([] (const auto& exp) {
                return mbind(exp, bookmark_from_json);
            })

            | sink([] (const auto &message) {
                const auto exp_bookmark = message.value;

                if (!exp_bookmark) {
                    message.reply("BD: danie niezrozumiae.\n");
                    return;
                }

                if (exp_bookmark->text.find("C++") != std::string::npos) {
                    message.reply("OK: " + to_string(exp_bookmark.get()) + "\n");
                } else {
                    message.reply("BD: odnonik nie dotyczy C++.\n");
                }
            });

	// Wykonanie ewaluacji wszystkich elementw zawartych w zakresie
    pipeline | to_vector;
}
