#ifndef EXPECTED_H
#define EXPECTED_H

// W oparciu o typ expected<T> zaprojektowany przez Alexandrescu,
// z dodatkiem pewnego lukru skadniowego.

template<typename T, typename E>
class expected {
protected:
    union {
        T m_value;
        E m_error;
    };

    bool m_isValid;

    expected() // uywane wewntrzne
    {
    }

public:
    ~expected()
    {
        if (m_isValid) {
            m_value.~T();
        } else {
            m_error.~E();
        }
    }

    expected(const expected &other)
        : m_isValid(other.m_isValid)
    {
        if (m_isValid) {
            new (&m_value) T(other.m_value);
        } else {
            new (&m_error) E(other.m_error);
        }
    }

    expected(expected &&other)
        : m_isValid(other.m_isValid)
    {
        if (m_isValid) {
            new (&m_value) T(std::move(other.m_value));
        } else {
            new (&m_error) E(std::move(other.m_error));
        }
    }

    expected &operator= (expected other)
    {
        swap(other);
        return *this;
    }

    void swap(expected &other)
    {
        using std::swap;
        if (m_isValid) {
            if (other.m_isValid) {
                // Oba poprawne, po prostu zamie wartoci.
                swap(m_value, other.m_value);

            } else {
                // Po naszej stronie wszystko OK, ale po drugiej jest problem.
				// Musimy wykona ca procedur.
                auto temp = std::move(other.m_error);       // przeniesienie bdu do zmiennej temp
                other.m_error.~E();                         // zniszczenie oryginalnego obiektu bdu
                new (&other.m_value) T(std::move(m_value)); // przeniesienie naszej wartoci do zmiennej other
                m_value.~T();                               // zniszczenie naszego obiektu value
                new (&m_error) E(std::move(temp));          // przeniesienie bdu zapisanego w temp do nas
                std::swap(m_isValid, other.m_isValid);      // zamiana flag isValid
            }

        } else {
            if (other.m_isValid) {
                // Po naszej stronie problem, po drugiej jest OK.
				// Wywoaj funkcj swap z danymi drugiej strony i skorzystaj
				// z implementacji poprzedniego przypadku.
                other.swap(*this);

            } else {
                // Wszdzie problemy, po prostu zamie bdy.
                swap(m_error, other.m_error);
                std::swap(m_isValid, other.m_isValid);
            }
        }
    }

    template <typename... ConsParams>
    static expected success(ConsParams && ...params)
    {
        expected result;
        result.m_isValid = true;
        new(&result.m_value) T(std::forward<ConsParams>(params)...);
        return result;
    }

    template <typename... ConsParams>
    static expected error(ConsParams && ...params)
    {
        expected result;
        result.m_isValid = false;
        new(&result.m_error) E(std::forward<ConsParams>(params)...);
        return result;
    }

    operator bool() const
    {
        return m_isValid;
    }

    bool is_valid() const
    {
        return m_isValid;
    };



    #ifdef NO_EXCEPTIONS
    #    define THROW_IF_EXCEPTIONS_ARE_ENABLED(WHAT) std::terminate()
    #else
    #    define THROW_IF_EXCEPTIONS_ARE_ENABLED(WHAT) throw std::logic_error(WHAT)
    #endif

    T &get()
    {
        if (!m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("expected<T, E> contains no value");
        return m_value;
    }

    const T &get() const
    {
        if (!m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("expected<T, E> contains no value");
        return m_value;
    }



    T *operator-> ()
    {
        return &get();
    }

    const T *operator-> () const
    {
        return &get();
    }

    E &error()
    {
        if (m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("There is no error in this expected<T, E>");
        return m_error;
    }

    const E &error() const
    {
        if (m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("There is no error in this expected<T, E>");
        return m_error;
    }

    #undef THROW_IF_EXCEPTIONS_ARE_ENABLED

    template <typename F>
    void visit(F f) {
        if (m_isValid) {
            f(m_value);
        } else {
            f(m_error);
        }
    }
};


template<typename E>
class expected<void, E> {
private:
    union {
        void* m_value;
        E m_error;
    };

    bool m_isValid;

    expected() {} // uywane wewntrznie

public:
    ~expected()
    {
        if (m_isValid) {
            // m_value.~T();
        } else {
            m_error.~E();
        }
    }

    expected(const expected &other)
        : m_isValid(other.m_isValid)
    {
        if (m_isValid) {
            // new (&m_value) T(other.m_value);
        } else {
            new (&m_error) E(other.m_error);
        }
    }

    expected(expected &&other)
        : m_isValid(other.m_isValid)
    {
        if (m_isValid) {
            // new (&m_value) T(std::move(other.m_value));
        } else {
            new (&m_error) E(std::move(other.m_error));
        }
    }

    expected &operator= (expected other)
    {
        swap(other);
        return *this;
    }

    void swap(expected &other)
    {
        using std::swap;
        if (m_isValid) {
            if (other.m_isValid) {
                // Obie strony poprawne, ale nie mamy adnych wartoci do zamiany.

            } else {
                // Po naszej stronie wszystko OK, ale po drugiej jest problem.
				// Musimy przenie bd do nas.
                auto temp = std::move(other.m_error);    // przenoszenie bdu do zmiennej temp
                other.m_error.~E();                      // niszczenie pierwotnego obiektu bdu
                new (&m_error) E(std::move(temp));       // przenoszenie bdu do nas
                std::swap(m_isValid, other.m_isValid);   // zamiania flag isValid
            }

        } else {
            if (other.m_isValid) {
                // Po naszej stronie problem, po drugiej jest OK.
				// Wywoaj funkcj swap z danymi drugiej strony i skorzystaj
				// z implementacji poprzedniego przypadku.
                other.swap(*this);

            } else {
                // Wszdzie problemy, po prostu zamie bdy.
                swap(m_error, other.m_error);
                std::swap(m_isValid, other.m_isValid);
            }
        }
    }

    static expected success()
    {
        expected result;
        result.m_isValid = true;
        result.m_value = nullptr;
        return result;
    }

    template <typename... ConsParams>
    static expected error(ConsParams && ...params)
    {
        expected result;
        result.m_isValid = false;
        new(&result.m_error) E(std::forward<ConsParams>(params)...);
        return result;
    }

    operator bool() const
    {
        return m_isValid;
    }

    bool is_valid() const
    {
        return m_isValid;
    };

    #ifdef NO_EXCEPTIONS
    #    define THROW_IF_EXCEPTIONS_ARE_ENABLED(WHAT) std::terminate()
    #else
    #    define THROW_IF_EXCEPTIONS_ARE_ENABLED(WHAT) throw std::logic_error(WHAT)
    #endif

    E &error()
    {
        if (m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("W obiekcie expected<T, E> nie ma bdu");
        return m_error;
    }

    const E &error() const
    {
        if (m_isValid) THROW_IF_EXCEPTIONS_ARE_ENABLED("W obiekcie expected<T, E> nie ma bdu");
        return m_error;
    }

};

template < typename T
         , typename E
         , typename Function
         , typename ResultType = decltype(std::declval<Function>()(std::declval<T>()))
         >
ResultType mbind(const expected<T, E>& exp, Function f)
{
    if (exp) {
        return std::invoke(f, exp.get());
    } else {
        return ResultType::error(exp.error());
    }
}

#endif

