(ns clojurebreaker.game
  (:use clojure.pprint)
  (:require [clojure.data :as data]
            [clojure.math.combinatorics :as comb]
            [clojure.java.io :as io]))

;; START:exact-matches
(defn exact-matches
  "Przyjmuje dwie kolekcje i zwraca liczb pozycji, na ktrych
   kolekcje obejmuj identyczne elementy."
  [c1 c2]
  (let [[_ _ matches] (data/diff c1 c2)]
    (count (remove nil? matches))))
;; END:exact-matches

;; START:unordered-matches
(defn unordered-matches
  "Przyjmuje dwie kolekcje i zwraca odwzorowanie, w ktrym kady klucz to
   element wystpujcy w obu kolekcjach, a kada warto to liczba wystpie
   elementu w kolekcji, gdzie dany element wystpuje mniejsz liczb razy."
  [c1 c2]
  (let [f1 (select-keys (frequencies c1) c2)
        f2 (select-keys (frequencies c2) c1)]
    (merge-with min f1 f2)))
;; END:unordered-matches

;; START:score
(defn score
  [c1 c2]
  (let [exact (exact-matches c1 c2)
        unordered (apply + (vals (unordered-matches c1 c2)))]
    {:exact exact :unordered (- unordered exact)}))
;; END: score

;; START: generate-turn-inputs
(defn generate-turn-inputs
  "Generuje wszystkie moliwe dane wejciowe dla gry Clojurebreaker
   dla kolorw colors i n kolumn."
  [colors n]
  (-> (comb/selections colors n)
      (comb/selections 2)))
;; END: generate-turn-inputs

;; START: score-inputs
(defn score-inputs
  "Na podstawie danych wejciowych zwraca leniw sekwencj odwzorowa
   z polami :secret, :guess i :score."
  [inputs]
  (map
   (fn [[secret guess]]
     {:secret (seq secret)
      :guess (seq guess)
      :score (score secret guess)})
   inputs))
;; END: score-inputs

(->> (generate-turn-inputs [:r :g :b] 2)
 (score-inputs))

;; Krok 17 - kolekcja score-table
#_(score-all-games [:R :G :B] 3)

;; Krok 18 - mona zapisa plik clj lub tabel z funkcji score-all-games
;; w systemie kontroli kodu rdowego i wykorzysta w testach regresji
;; (naley doda clojure.java.io)
(use 'clojure.pprint)
(with-open [w (io/writer "scoring-table")]
    (binding [*out* w]
      (print-table (->> (generate-turn-inputs [:r :g :b :y] 4)
                        (score-inputs)))))
