// Rysunek 24.25. ResultSetTableModel.java
// Implementacja TableModel, która dostarcza ResultSet do obiektu JTable
import java.sql.Connection;
import java.sql.Statement;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import javax.swing.table.AbstractTableModel;

// W ResultSet wiersze i kolumny są liczone od 1, a w JTable 
// wiersze i kolumny są liczone od 0. Przetwarzając
// wiersze i kolumny ResultSet w celu użycia w JTable, konieczne
// jest dodanie 1 do każdego numeru wiersza i kolumny,
// aby pobierać właściwe dane z ResultSet (czyli kolumna 0 z JTable
// to kolumna 1 z ResultSet i wiersz 0 z JTable to wiersz 1 z ResultSet).
public class ResultSetTableModel extends AbstractTableModel {
   private final Connection connection;
   private final Statement statement;
   private ResultSet resultSet;
   private ResultSetMetaData metaData;
   private int numberOfRows;

   // Śledzenie stanu połączenia z bazą danych
   private boolean connectedToDatabase = false;
   
   // Konstruktor inicjalizuje resultSet i pobiera obiekt metadanych;
   // określa również liczbę wierszy
   public ResultSetTableModel(String url, String username,
       String password, String query) throws SQLException {
      // Połączenie z bazą danych.
      connection = DriverManager.getConnection(url, username, password);

      // Utworzenie Statement w celu odpytania bazy danych                    
      statement = connection.createStatement(                           
         ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);

      // Aktualizacja statusu połączenia z bazą danych
      connectedToDatabase = true;         

      // Ustawienie zapytania i wykonanie go
      setQuery(query);
   } 

   // Pobranie klasy, która reprezentuje typ kolumny
   public Class getColumnClass(int column) throws IllegalStateException {
      // Upewnij się, że połączenie z bazą danych jest dostępne                   
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      }                                                               

      // Określ klasę Javy dla kolumny
      try {
         String className = metaData.getColumnClassName(column + 1);
         
         // Zwróć obiekt Class reprezentujący className
         return Class.forName(className);              
      } 
      catch (Exception exception) {
         exception.printStackTrace();
      } 
      
      return Object.class; // Jeśli pojawią się problemy, załóż typ Object
   } 

   // Pobierz liczbę kolumn w ResultSet
   public int getColumnCount() throws IllegalStateException {
      // Upewnij się, że połączenie z bazą danych jest dostępne                    
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      }                                                               

      // Określ liczbę kolumn
      try {
         return metaData.getColumnCount(); 
      } 
      catch (SQLException sqlException) {
         sqlException.printStackTrace();
      } 
      
      return 0; // Jeśli pojawią się problemy, zwróć 0 jako liczbę kolumn
   } 

   // Pobierz nazwę konkretnej kolumny z ResultSet
   public String getColumnName(int column) throws IllegalStateException {
      // Upewnij się, że połączenie z bazą danych jest dostępne                     
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      }                                                               

      // Określ nazwę kolumny
      try {
         return metaData.getColumnName(column + 1);  
      } 
      catch (SQLException sqlException) {
         sqlException.printStackTrace();
      } 
      
      return ""; // Jeśli pojawią się problemy, zwróć pusty tekst jako nazwę kolumny
   } 

   // Zwróć liczbę wierszy w ResultSet
   public int getRowCount() throws IllegalStateException {
      // Upewnij się, że połączenie z bazą danych jest dostępne                   
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      }                                                               
 
      return numberOfRows;
   } 

   // Pobierz wartość w konkretnym wierszu i kolumnie
   public Object getValueAt(int row, int column) 
      throws IllegalStateException {

      // Upewnij się, że połączenie z bazą danych jest dostępne                    
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      } 

      // Pobierz wartość z konkretnego wiersza i kolumny ResultSet
      try {
         resultSet.absolute(row + 1);           
         return resultSet.getObject(column + 1);
      } 
      catch (SQLException sqlException) {
         sqlException.printStackTrace();
      } 
      
      return ""; // Jeśli pojawią się problemy, zwróć pusty tekst
   } 
   
   // Ustaw nowe zapytanie do bazy danych
   public void setQuery(String query) 
      throws SQLException, IllegalStateException {
   
      // Upewnij się, że połączenie z bazą danych jest dostępne                     
      if (!connectedToDatabase) {                                     
         throw new IllegalStateException("Brak połączenia z bazą danych");
      } 

      // Określ zapytanie i wykonaj je
      resultSet = statement.executeQuery(query);

      // Pobierz metadane z ResultSet
      metaData = resultSet.getMetaData(); 

      // Określ liczbę wierszy w ResultSet
      resultSet.last(); // Przejdź do ostatniego wiersza
      numberOfRows = resultSet.getRow(); // Pobierz numer wiersza    
      
      
      fireTableStructureChanged(); // Poinformuj JTable, że model uległ zmianie
   } 
                                                  
   // Zamknięcie Statement i Connection         
   public void disconnectFromDatabase() {             
      if (connectedToDatabase) {                      
         // Zamknij Statement i Connection        
         try {                                        
            resultSet.close();                        
            statement.close();                        
            connection.close();                       
         }                                            
         catch (SQLException sqlException) {          
            sqlException.printStackTrace();           
         }                                            
         finally { // Uaktualnij status połączenia z bazą danych
            connectedToDatabase = false;              
         }                      
      }                    
   }              
}





/**************************************************************************
 * (C) Copyright 1992-2018  by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/
