// Rysunek 23.20. SynchronizedBuffer.java
// Synchronizacja dostępu do współdzielonej wartości typu int 
// za pomocą obiektów Lock i Condition
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.Condition;

public class SynchronizedBuffer implements Buffer {
   // Obiekt Lock pozwalający na synchronizację dostępu do bufora
   private final Lock accessLock = new ReentrantLock();

   // Warunki sterujące odczytem i zapisem
   private final Condition canWrite = accessLock.newCondition();
   private final Condition canRead = accessLock.newCondition(); 

   private int buffer = -1; // Współdzielone przez wątki producenta i konsumenta
   private boolean occupied = false; // Informacja, czy bufor jest zajęty
   
   // Umieszczenie wartości int w buforze
   @Override
   public void blockingPut(int value) throws InterruptedException {
      accessLock.lock(); // Zablokuj obiekt

      // Wyświetlenie informacji o wątku i buforze, a następnie oczekiwanie
      try {
         // Jeśli bufor nie jest pusty, umieść wątek w stanie oczekiwania
         while (occupied) {
            System.out.println("Producer próbuje zapisu.");
            displayState("Buffer jest pełny. Producer czeka.");
            canWrite.await(); // Czekaj, aż bufor będzie pusty
         } 

         buffer = value; // Ustaw nową wartość bufora

         // Wskaż, że producent nie może zapamiętać innej wartości,
         // dopóki konsument nie pobierze aktualnej
         occupied = true;
        
         displayState("Producer zapisuje " + buffer);

         // Poinformuj inne wątki, że mogą odczytywać
         canRead.signalAll();                             
      } 
      finally {
         accessLock.unlock(); // Odblokuj obiekt
      }
   } 
    
   // Odczyt wartości z bufora
   @Override
   public int blockingGet() throws InterruptedException {
      int readValue = 0; // Zainicjalizuj wartość odczytywaną z bufora
      accessLock.lock(); // Zablokuj obiekt

      // Wyświetlenie informacji o wątku i buforze, a następnie oczekiwanie
      try {
         // Jeśli nie ma danych do odczytania, umieść wątek w stanie oczekiwania
         while (!occupied) {
            System.out.println("Consumer próbuje odczytu.");
            displayState("Buffer jest pusty. Consumer czeka.");
            canRead.await(); // Czekaj, aż bufor się zapełni
         } 

         // Wskaż, że producent może zapisać nową wartość, bo
         // konsument pobrał aktualną
         occupied = false;

         readValue = buffer; // Pobierz wartość z bufora
         displayState("Consumer odczytuje " + readValue);

         // Poinformuj inne wątki, że bufor jest pusty
         canWrite.signalAll();                               
      } 
      finally {
         accessLock.unlock(); // Odblokuj obiekt
      } 

      return readValue;
   } 
    
   // Wyświetl aktualną operację i stan bufora
   private void displayState(String operation) {
      try {
         accessLock.lock(); // Zablokuj obiekt
         System.out.printf("%-40s%d\t\t%b%n%n", operation, buffer, 
            occupied);
      }
      finally {
         accessLock.unlock(); // Odblokuj obiekt
      }
   }
}



/**************************************************************************
 * (C) Copyright 1992-2015 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/