// Rysunek 23.31. FibonacciDemo.java
// Obliczanie elementów ciągu Fibonacciego synchronicznie i asynchronicznie
import java.time.Duration;
import java.text.NumberFormat;
import java.time.Instant;
import java.util.concurrent.CompletableFuture; 
import java.util.concurrent.ExecutionException;

// Klasa, która przechowuje dwie wartości typu Instant
class TimeData {
   public Instant start;
   public Instant end;

   // Łączny czas w sekundach
   public double timeInSeconds() {
      return Duration.between(start, end).toMillis() / 1000.0;
   }
} 

public class FibonacciDemo {
   public static void main(String[] args) 
      throws InterruptedException, ExecutionException {
   
      // Wykonaj synchroniczne obliczenia fibonacci(45) i fibonacci(44)
      System.out.println("Synchroniczne i długo wykonywane działania");
      TimeData synchronousResult1 = startFibonacci(45);
      TimeData synchronousResult2 = startFibonacci(44);
      double synchronousTime = 
         calculateTime(synchronousResult1, synchronousResult2);
      System.out.printf(
         "  Łączny czas obliczeń = %.3f sekund%n", synchronousTime);

      // Wykonaj asynchroniczne obliczenia fibonacci(45) i fibonacci(44)
      System.out.printf("%nAsynchroniczne i długo wykonywane działania%n");
      CompletableFuture<TimeData> futureResult1 =                
         CompletableFuture.supplyAsync(() -> startFibonacci(45));
      CompletableFuture<TimeData> futureResult2 =                
         CompletableFuture.supplyAsync(() -> startFibonacci(44));

      // Oczekiwanie na wyniki z asynchronicznych operacji
      TimeData asynchronousResult1 = futureResult1.get();
      TimeData asynchronousResult2 = futureResult2.get();
      double asynchronousTime = 
         calculateTime(asynchronousResult1, asynchronousResult2);
      System.out.printf(
         "  Łączny czas obliczeń = %.3f sekund%n", asynchronousTime);

      // Wyświetlenie różnicy czasu jako procentu
      String percentage = NumberFormat.getPercentInstance().format(
         (synchronousTime - asynchronousTime) / asynchronousTime);
      System.out.printf("%nObliczenia synchroniczne zajęły %s" +
         " więcej czasu niż asynchroniczne%n", percentage);
   }

   // Wykonuje funkcję fibonacci asynchronicznie
   private static TimeData startFibonacci(int n) {
      // Utwórz obiekt TimeData w celu przechowania czasów
      TimeData timeData = new TimeData();
    
      System.out.printf("  Obliczanie fibonacci(%d)%n", n);
      timeData.start = Instant.now(); 
      long fibonacciValue = fibonacci(n);
      timeData.end = Instant.now(); 
      displayResult(n, fibonacciValue, timeData);
      return timeData;
   } 

   // Rekurencyjna metoda fibonacci; oblicza n-tą wartość ciągu Fibonacciego
   private static long fibonacci(long n) {
      if (n == 0 || n == 1) {
         return n;
      }
      else {
         return fibonacci(n - 1) + fibonacci(n - 2);
      }
   } 

   // Wyświetla wyniki i łączny czas działania
   private static void displayResult(
      int n, long value, TimeData timeData) {

      System.out.printf("  fibonacci(%d) = %d%n", n, value);
      System.out.printf(
         "  Czas obliczeń dla fibonacci(%d) = %.3f sekund%n", 
         n, timeData.timeInSeconds()); 
   }

   // Oblicza czas w sekundach
   private static double calculateTime(
      TimeData result1, TimeData result2) {

      TimeData bothThreads = new TimeData();

      // Znajdź czas wcześniejszy
      bothThreads.start = result1.start.compareTo(result2.start) < 0 ? 
         result1.start : result2.start;

      // Znajdź czas późniejszy
      bothThreads.end = result1.end.compareTo(result2.end) > 0 ? 
         result1.end : result2.end;

      return bothThreads.timeInSeconds();
   }
} 

/**************************************************************************
 * (C) Copyright 1992-2015 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/

