// Rysunek 17.14. ProcessingEmployees.java
// Przetwarzanie strumienia obiektów Employee
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;               
import java.util.TreeMap;           
import java.util.function.Function; 
import java.util.function.Predicate;
import java.util.stream.Collectors; 

public class ProcessingEmployees {
   public static void main(String[] args) {
      // Inicjalizacja tablicy Employees
      Employee[] employees = {
         new Employee("Jan", "Kowalski", 5000, "IT"),
         new Employee("Anna", "Kusar", 7600, "IT"),
         new Employee("Marcin", "Nowak", 3587.5, "Sprzedaż"),
         new Employee("Jerzy", "Nowak", 4700.77, "Marketing"),
         new Employee("Lech", "Nowak", 6200, "IT"),
         new Employee("Jan", "Zielony", 3200, "Sprzedaż"),
         new Employee("Wanda", "Kapusta", 4236.4, "Marketing")};

      // Pobierz widok List dla obiektów Employee
      List<Employee> list = Arrays.asList(employees);

      // Wyświetl wszystkich pracowników
      System.out.println("Kompletna lista pracowników:");
      list.stream().forEach(System.out::println);
      
      // Predykat, który zwraca true dla wynagrodzeń z przedziału od 4000 do 6000 zł
      Predicate<Employee> fourToSixThousand =                  
         e -> (e.getSalary() >= 4000 && e.getSalary() <= 6000);

      // Wyświetl pracowników z wynagrodzeniami z przedziału od 4000 do 6000 zł
      // posortowanych w kolejności rosnącej od najniższego wynagrodzenia
      System.out.printf(
         "%nPracownicy zarabiający od 4000 do 6000 zł miesięcznie posortowani według wysokości wynagrodzenia:%n");
      list.stream()                                         
          .filter(fourToSixThousand)                        
          .sorted(Comparator.comparing(Employee::getSalary))
          .forEach(System.out::println);                    

      // Wyświetl pierwszego pracownika otrzymującego wynagrodzenie z przedziału od 4000 do 6000 zł
      System.out.printf("%nPierwszy pracownik zarabiający od 4000 do 6000 zł:%n%s%n",
         list.stream()                 
             .filter(fourToSixThousand)
             .findFirst()
             .get());

      // Funkcje pobierające imię i nazwisko pracownika
      Function<Employee, String> byFirstName = Employee::getFirstName;
      Function<Employee, String> byLastName = Employee::getLastName;  

      // Komparator porównujący pracowników najpierw po nazwiskach, a potem po imionach
      Comparator<Employee> lastThenFirst =                           
         Comparator.comparing(byLastName).thenComparing(byFirstName);

      // Posortuj pracowników rosnąco po nazwisku, a potem po imieniu
      System.out.printf(
         "%nPracownicy posortowani rosnąco po nazwisku i imieniu:%n");
      list.stream()
          .sorted(lastThenFirst)
          .forEach(System.out::println);

      // Posortuj pracowników malejąco po nazwisku, a potem po imieniu
      System.out.printf(
         "%nPracownicy posortowani malejąco po nazwisku i imieniu:%n");
      list.stream()
          .sorted(lastThenFirst.reversed())
          .forEach(System.out::println);

      // Wyświetl posortowane unikatowe nazwiska pracowników
      System.out.printf("%nUnikatowe nazwiska pracowników:%n");
      list.stream()
          .map(Employee::getLastName)
          .distinct()                
          .sorted()
          .forEach(System.out::println);

      // Wyświetlenie jedynie imion i nazwisk pracowników
      System.out.printf(
         "%nPracownicy posortowani rosnąco po nazwisku i imieniu:%n"); 
      list.stream()
          .sorted(lastThenFirst)
          .map(Employee::getName)
          .forEach(System.out::println);

      // Pogrupuj pracowników według działów
      System.out.printf("%nPracownicy pogrupowani według działów:%n");
      Map<String, List<Employee>> groupedByDepartment =               
         list.stream()                                                
             .collect(Collectors.groupingBy(Employee::getDepartment));
      groupedByDepartment.forEach(                                    
         (department, employeesInDepartment) -> {                     
            System.out.printf("%n%s%n", department);                  
            employeesInDepartment.forEach(                            
               employee -> System.out.printf("   %s%n", employee));   
         }                                                            
      );                                                              

      // Zlicz pracowników w każdym dziale
      System.out.printf("%nLiczba pracowników w każdym dziale:%n");
      Map<String, Long> employeeCountByDepartment =                 
         list.stream()                                              
             .collect(Collectors.groupingBy(Employee::getDepartment,
                Collectors.counting()));                            
      employeeCountByDepartment.forEach(                            
         (department, count) -> System.out.printf(                  
            "%s ma %d pracowników%n", department, count));         

      // Suma wynagrodzeń pracowników za pomocą metody sum z DoubleStream
      System.out.printf(
         "%nSuma zarobków pracowników (za pomocą metody sum): %.2f%n",
         list.stream()
             .mapToDouble(Employee::getSalary)
             .sum());

      // Suma wynagrodzeń pracowników za pomocą metody reduce ze Stream
      System.out.printf(
         "Suma zarobków pracowników (za pomocą metody reduce): %.2f%n",
         list.stream()
             .mapToDouble(Employee::getSalary)              
             .reduce(0, (value1, value2) -> value1 + value2));  

      // Średnia wynagrodzeń pracowników za pomocą metody average z DoubleStream
      System.out.printf("Średnia zarobków pracowników: %.2f%n",
         list.stream()
             .mapToDouble(Employee::getSalary)
             .average()
             .getAsDouble());      
   } 
} 


 
 /*************************************************************************
 * (C) Copyright 1992-2018 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/
