/******************************************************************************
 *  Kompilacja:  javac PathFinder.java
 *  Wykonanie:    java Pathfinder input.txt delimiter source
 *  Zależności: Queue.java Stack.java Graph.java
 *  Pliki z danymi:   https://introcs.cs.princeton.edu/45graph/routes.txt
 *                https://introcs.cs.princeton.edu/45graph/movies.txt
 *  
 *  Uruchamia algorytm przeszukiwania wszerz ze źródłem w wierzchołku s w grafie G.
 *  Po wstępnym przetworzeniu grafu potrafi zwracać najkrótsze ścieżki
 *  z s do dowolnego wierzchołka t.
 *
 *  % java PathFinder routes.txt " " JFK
 *  LAX
 *     JFK
 *     ORD
 *     PHX
 *     LAX
 *  odległość 3
 *  MCO
 *     JFK
 *     MCO
 *  odległość 1
 *  DFW
 *     JFK
 *     ORD
 *     DFW
 *  odległość 2
 *
 ******************************************************************************/

public class PathFinder {

    // prev[v] = poprzedni wierzchołek na najkrótszej ścieżce z s do v
    // dist[v] = długość najkrótszej ścieżki z s do v
    private ST<String, String>  prev = new ST<String, String>();
    private ST<String, Integer> dist = new ST<String, Integer>();

    // uruchamia przeszukiwanie wszerz w grafie G dla danego wierzchołka źródłowego s
    public PathFinder(Graph G, String s) {

        // umieszczanie wierzchołka źródłowego w kolejce
        Queue<String> queue = new Queue<String>();
        queue.enqueue(s);
        dist.put(s, 0);
        
        // wielokrotne usuwanie następnego wierzchołka v z kolejki i wstawianie
        // wszystkich jego sąsiadów (jeśli nie zostali jeszcze odwiedzeni)
        while (!queue.isEmpty()) {
            String v = queue.dequeue();
            for (String w : G.adjacentTo(v)) {
                if (!dist.contains(w)) {
                    queue.enqueue(w);
                    dist.put(w, 1 + dist.get(v));
                    prev.put(w, v);
                }
            }
        }
    }

    // czy v jest osiągalny ze źródła s?
    public boolean hasPathTo(String v) {
        return dist.contains(v);
    }

    // zwracanie długości najkrótszej ścieżki z v do s
    public int distanceTo(String v) {
        if (!hasPathTo(v)) return Integer.MAX_VALUE;
        return dist.get(v);
    }

    // zwracanie najkrótszej ścieżki z v do s jako obiektu typu Iterable
    public Iterable<String> pathTo(String v) {
        Stack<String> path = new Stack<String>();
        while (v != null && dist.contains(v)) {
            path.push(v);
            v = prev.get(v);
        }
        return path;
    }


    public static void main(String[] args) {
        String filename  = args[0];
        String delimiter = args[1];
        Graph G = new Graph(filename, delimiter);
        String s = args[2];
        PathFinder pf = new PathFinder(G, s);
        while (!StdIn.isEmpty()) {
            String t = StdIn.readLine();
            for (String v : pf.pathTo(t)) {
                StdOut.println("   " + v);
            }
            StdOut.println("odległość " + pf.distanceTo(t));
        }
    }


}