/**
  Listingi z rozdziału 5.
  Autor: Luis Atencio
*/
"use strict";

QUnit.module('Rozdział 5.');

// Biblioteki funkcyjne używane w tym rozdziale
const _ = require('lodash');
const R = require('ramda');

// Używane monady i funktory
const Wrapper = require('../model/Wrapper.js').Wrapper;
const wrap = require('../model/Wrapper.js').wrap;
const empty = require('../model/Empty.js').empty;
const Maybe = require('../model/monad/Maybe.js').Maybe;
const Either = require('../model/monad/Either.js').Either;

// Używane modele
const Student = require('../model/Student.js').Student;
const Address = require('../model/Address.js').Address;
const Person = require('../model/Person.js').Person;

QUnit.test("Prosty test nakładki", function () {
	const wrappedValue = wrap('Pracuj w stylu funkcyjnym');
	assert.equal(wrappedValue.map(R.identity), 'Pracuj w stylu funkcyjnym'); //-> 'Pracuj w stylu funkcyjnym'
});
 
QUnit.test("Prosty test funktora", function () {
	const plus = R.curry((a, b) => a + b);
	const plus3 = plus(3);
	const plus10 = plus(10);
	const two = wrap(2);
	const five = two.fmap(plus3); //-> Wrapper(5)
	assert.equal(five.map(R.identity), 5); //-> 5

	assert.equal(two.fmap(plus3).fmap(plus10).map(R.identity), 15); //-> Wrapper(15)
});

QUnit.test("Prosta funkcja z nakładką", function () {
	// Używanie pomocniczego obiektu db z rozdziału 1.	
	const db = require('../ch01/helper').db;	

	const find = R.curry((db, id) => db.find(id));

	const findStudent = R.curry((db, ssn) => {
		return wrap(find(db, ssn));
	});
	
	const getAddress = (student) => {
		return wrap(student.fmap(R.prop('firstname')));
	};

	const studentAddress = R.compose(
		getAddress,
		findStudent(db)
	);

	assert.deepEqual(studentAddress('444-44-4444'), wrap(wrap('Alonzo')));
});

QUnit.test("Prosty pusty kontener", function () {
	
	const isEven = (n) => Number.isFinite(n) && (n % 2 == 0);
	const half = (val) => isEven(val) ? wrap(val / 2) : empty();
	assert.deepEqual(half(4), wrap(2)); //-> Wrapper(2)
	assert.deepEqual(half(3), empty()); //-> Empty	
});


QUnit.test("Prosty kontener", function () {
	const WrapperMonad = require('../model/monad/Wrapper.js').Wrapper;
	
	let result = WrapperMonad.of('Witajcie, monady!')
		.map(R.toUpper)
		.map(R.identity); //-> Wrapper('WITAJCIE, MONADY!')
 	
 	assert.deepEqual(result, new WrapperMonad('WITAJCIE, MONADY!'));
});

QUnit.test("Prosty test monady Maybe", function () {	
	let result = Maybe.of('Witaj, Maybe!').map(R.toUpper);
 	assert.deepEqual(result, Maybe.of('WITAJ, MAYBE!'));

	const Nothing = require('../model/monad/Maybe.js').Nothing;
 	result = Maybe.fromNullable(null);
 	assert.deepEqual(result, new Nothing(null));
});

QUnit.test("Używanie monady Maybe do pobierania zagnieżdżonej właściwości", function () {	

	let address = new Address('US');
	let student = new Student('444-44-4444', 'Joe', 'Smith', 
		'Harvard', 1960, address);

	const getCountry = (student) => student		
		.map(R.prop('address'))
		.map(R.prop('country'))
		.getOrElse('Kraj nie istnieje!');
	
	assert.equal(getCountry(Maybe.fromNullable(student)), address.country);
});

QUnit.test("Używanie monady Maybe do pobierania nieistniejącej zagnieżdżonej właściwości", function () {	
	
	let student = new Student('444-44-4444', 'Joe', 'Smith', 
		'Harvard', 1960, null);

	const getCountry = (student) => student		
		.map(R.prop('address'))
		.map(R.prop('country'))
		.getOrElse('Kraj nie istnieje!');
	
	assert.equal(getCountry(Maybe.fromNullable(student)), 'Kraj nie istnieje!');
});


QUnit.test("Prosty test monady Either", function () {	

	// Używanie pomocniczego obiektu db z rozdziału 1.	
	const db = require('../ch01/helper').db;	

	const find = R.curry((db, id) => db.find(id));

	const Left = require('../model/monad/Either.js').Left;

	const safeFindObject = R.curry(function (db, id) {
		const obj = find(db, id);
		if(obj) {
			return Either.of(obj);
		}
		return Either.left(`Nie znaleziono obiektu o ID: ${id}`);
	});

	const findStudent = safeFindObject(db);
	let result = findStudent('444-44-4444').getOrElse(new Student());
	assert.deepEqual(result, new Person('444-44-4444', 'Alonzo', 'Church'));

	result = findStudent('xxx-xx-xxxx');
	assert.deepEqual(result, Either.left(`Nie znaleziono obiektu o ID: xxx-xx-xxxx`));	

	assert.throws(() => {
		console.log(result.value);
	}, TypeError);
});

// Kod używany w dalszych testach jednostkowych

// Używanie pomocniczego obiektu db z rozdziału 1.	
const db = require('../ch01/helper').db;	

// validLength :: Number, String -> Boolean
const validLength = (len, str) => str.length === len;

const find = R.curry((db, id) => db.find(id));

// checkLengthSsn :: String -> Either(String)
const checkLengthSsn = ssn => {		
	return Either.of(ssn)
		.filter(R.partial(validLength, [9]));
};

// safeFindObject :: Store, string -> Either(Object)
const safeFindObject = R.curry((db, id) => Either.fromNullable(find(db, id)));

// finStudent :: String -> Either(Student)
const findStudent = safeFindObject(db);

// csv :: Array => String
const csv = arr => arr.join(',');

const trim = (str) => str.replace(/^\s*|\s*$/g, '');
const normalize = (str) => str.replace(/\-/g, '');
const cleanInput = R.compose(normalize, trim);

QUnit.test("Używanie monady Either w programie showStudent", function () {	

	const showStudent = (ssn) =>
		Maybe.fromNullable(ssn)
			.map(cleanInput)
			.chain(checkLengthSsn)
		 	.chain(findStudent)
		    .map(R.props(['ssn', 'firstname', 'lastname']))
		    .map (csv)
			.map (R.tap(console.log));  //-> Używanie R.tap do symulowania efektów ubocznych (w książce dane są dodawane do modelu DOM)

	let result = showStudent('444-44-4444').getOrElse('Studenta nie znaleziono!')
	assert.equal(result, '444-44-4444,Alonzo,Church');

	result = showStudent('xxx-xx-xxxx').getOrElse('Studenta nie znaleziono!');
	assert.equal(result, 'Studenta nie znaleziono!');
});

QUnit.test("Monady z użyciem programowalnych przecinków", function () {	

	// map :: (ObjectA -> ObjectB), Monad -> Monad[ObjectB]
	const map = R.curry((f, container) => container.map(f));
	// chain :: (ObjectA -> ObjectB), M -> ObjectB
	const chain = R.curry((f, container) => container.chain(f));

	const lift = R.curry((f, obj) => Maybe.fromNullable(f(obj)));

	const trace = R.curry((msg, obj) => console.log(msg));

	const showStudent = R.compose(
		R.tap(trace('Student printed to the console')),
		map(R.tap(console.log)),   //-> Używanie R.tap do symulowania efektów ubocznych (w książce dane są dodawane do modelu DOM)

		R.tap(trace('Dane o studencie przekształcone na format CSV')),
		map(csv),

		map(R.props(['ssn', 'firstname', 'lastname'])),

		R.tap(trace('Rekord pobrany z powodzeniem!')),
		chain(findStudent),

		R.tap(trace('Dane wejściowe były poprawne')),
		chain(checkLengthSsn),
		lift(cleanInput)
		);

	let result = showStudent('444-44-4444').getOrElse('Studenta nie znaleziono!');
	assert.equal(result, '444-44-4444,Alonzo,Church');
});






