/**
  * Przykd dla podrozdziau "Szybko algorytmu" (implementacje
  * popularnych algorytmw sortujcych).
  */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "tests.h"

typedef unsigned long ELEMENT;

const int MAX_SIZE=10000000;
const int TEST_SIZE = 1000;

ELEMENT *array = 0;

ELEMENT *tmp_array;

#define SWAP(A,B) { ELEMENT tmp; tmp = A; A = B; B = tmp; }

/*
 * Prototyp uniwersalnej funkcji sortujcej
 */

typedef void (*SORT_FUNCTION)(ELEMENT *, int);

/*
 * Sortowanie przez wybieranie
 */

/* START: selection */
void selection(ELEMENT *input, int n) {
  int i, j;

  for (i = 0; i < n-1; i++) {
    int     lowest_index = i;
    ELEMENT lowest_value = input[i];

    for (j = i+1; j < n; j++)
      if (input[j] < lowest_value) {
	lowest_value = input[j];
	lowest_index = j;
      }

    SWAP(input[i], input[lowest_index]);
  }
}
/* END: selection */

/*
 * Sortowanie bbelkowe
 */

void bubble(ELEMENT *input, int n) {
  ELEMENT tmp;
  int i;

  do {
    
    tmp = input[0];

    for (i = 1; i < n; i++) {
      if (input[i-1] > input[i]) {
	tmp = input[i-1];
	input[i-1] = input[i];
	input[i] = tmp;
      }
    }
  } while (tmp != input[0]);
}


/*
 * Sortowanie przez wstawianie
 */

void insertion(ELEMENT *input, int n) {

  int i, j;
  ELEMENT tmp;

  for (i = 1; i < n; i++) {
    tmp = input[i];
    j = i;

    while ((j > 0) && (input[j-1] > tmp)) {
      input[j] = input[j-1];
      j--;
    }
    input[j] = tmp;
  }
}

/* 
 * Sortowanie szybkie
 */

/* START: quicksort */
int partition(ELEMENT *input, int first, int last) {
  ELEMENT pivot = input[first];
  int     right = last;
  int     left  = first+1;

  while (1) {

    while (input[right] > pivot) right--;
    while ((left < right) &&
	   input[left] <= pivot) left++;
    if (left >= right)
      break;
    SWAP(input[left], input[right]);
  }

  input[first] = input[right];
  input[right] = pivot;
  return right;
}

void quicksort(ELEMENT *input, int first, int last) {

  if (last > first) {
    int pivot_index = partition(input, first, last);
    quicksort(input, first, pivot_index-1);
    quicksort(input, pivot_index+1, last);
  }
}
/* END: quicksort */

void quick(ELEMENT *input, int count) {
  quicksort(input, 0, count-1);
}

/*
 * Sortowanie Shella
 */

void shell(ELEMENT *input, int n) {
  int      h = 1;
  ELEMENT  tmp;
  int      i, j;

  do { h = h*3+1; } while (h <= n);

  do {
    h = h/3;

    for (i = h; i < n; i++) {

      tmp = input[i];
      j = i;

      while (input[j-h]>tmp) {
	input[j] = input[j-h];
	j -= h;
	if (j < h)
	  break;
      }
      input[j] = tmp;
    }
  } while (h > 1);
}

/*
 * Sortowanie wedug cyfr (uwaga: tylko dla liczb bez znaku)
 */
/* START: radix */
void radix(ELEMENT *input, int n) {
  long count[256];

  int shift = 0;
  int i;

  while (shift < 32) {
    memset(count, 0, sizeof(count));

    for (i = 0; i < n; i++) {
      int bits = (input[i] >> shift) & 255;
      count[bits]++;
    }

    for (i = 1; i < 256; i++)
      count[i] += count[i-1];

    for (i = n-1; i >= 0; i--) {
      int bits = (input[i] >> shift) & 255;
      tmp_array[count[bits]] = input[i];
      count[bits]--;
    }

    memcpy(input, tmp_array, n*sizeof(input[0]));

    shift += 8;
  }
}
/* END: radix */

/*
 * Sprawdza, czy tablica jest posortowana; jeli nie, exit(1)
 */

void checkSorted(ELEMENT *array, int size, const char *type, const char *msg) {
  int i;

  for (i = 1; i < size; i++) {
    if (array[i-1] > array[i]) {
      fail("%s: bd w tecie '%s'", type, msg);
      exit(1);
    }
  }
}

/*
 * testSort - wywouje funkcj sortujc, po czym sprawdza porzdek elementw
 */

void testSort(SORT_FUNCTION fn, const char *name) {

  int j;

  /* inicjalizuje tablic przy uyciu wartoci losowych */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = random();

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "losowe wypenienie");

  /* inicjalizuje tablic przy uyciu tych samych liczb */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = 0;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "zera");

  /* inicjalizuje tablic przy uyciu rosncych wartoci */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = j;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "rosnce");

  /* inicjalizuje tablic przy uyciu malejcych wartoci */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = TEST_SIZE - j;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "malejce");
}

/*
 * runSort - uruchamia poszczeglne funkcje dla coraz wikszych tablic
 */

void runSort(SORT_FUNCTION fn, const char *name, 
	     int min, int max, int step, int test) {

  int howBig;

  /* zawsze zaczyna od tych samych danych */
  srandom(1);

  if (test) {
    testSort(fn, name);
  }
  else {

    howBig = min;
    
    do {
      int j;
      struct timeval start, end;
      double diff;
      
      /* inicjalizuje tablic liczb do posortowania */
      for (j = 0; j < howBig; j++)
	array[j] = random();
      
      gettimeofday(&start, 0);
      
      (*fn)(array, howBig);
      
      gettimeofday(&end, 0);
      
      diff = (end.tv_sec - start.tv_sec) + 
	(double)(end.tv_usec-start.tv_usec)/1e6;
      
      printf("%-10s %8d  %g\n", name, howBig, diff);
      
      howBig += step;
      
    } while (howBig <= max);
  }
}

/*
 * Sposb uruchamiania:
 *
 *    ./sort [-t] [name ...]
 *
 * gdzie 'name' to co najmniej jeden algorytm sortowania. Uycie opcji '-t' powoduje
 * testowanie poprawnoci sortowania (zamiast mierzenia czasu).
 *
 */

int main(int argc, char **argv) {

  int testFlag = 0;
  long arraySize = MAX_SIZE;

  argc--; argv++;

  if (argc && (strcmp(*argv, "-t") == 0)) {
    argc--;
    argv++;
    testFlag++;
    arraySize = TEST_SIZE;
  }

  /*
   * Aby poprawi wydajno, tablica jest alokowana z wyprzedzeniem.
   */

  array = (ELEMENT *)malloc(arraySize*sizeof(array[0]));

  if (array == 0) {
    fprintf(stderr, "Brak pamici dla tablicy\n");
    exit(1);
  }

  /*
   * Uruchamia funkcje sortujce.
   */
  while (argc--) {

    if (strcmp(*argv, "wg cyfr") == 0) {
      tmp_array = (ELEMENT *)malloc(arraySize*sizeof(array[0]));
      runSort(radix, "g cyfr", 1000000, MAX_SIZE, 1000000, testFlag);
      free(tmp_array);
    }
    else if (strcmp(*argv, "szybkie") == 0)
      runSort(quick, "szybkie", 1000000, MAX_SIZE, 1000000, testFlag);
    else if (strcmp(*argv, "przez wybieranie") == 0)
      runSort(selection, "przez wybieranie", 10000, 100000, 10000, testFlag);
    else if (strcmp(*argv, "Shella") == 0)
      runSort(shell, "Shella", 10000, MAX_SIZE, 10000, testFlag);
    else if (strcmp(*argv, "przez wstawianie") == 0)
      runSort(insertion, "przez wstawianie", 10000, 100000, 100000, testFlag);
    else if (strcmp(*argv, "bbelkowe") == 0)
      runSort(bubble, "bbelkowe", 10000, 100000, 100000, testFlag);
    else 
      fprintf(stderr, "Nieznany typ sortowania '%s'\n", *argv);

    argv++;
  }

  exit(0);
}
