/***
 * Fragment z książki „Pragmatyczny programista. Wydanie jubileuszowe”,
 * wydanej w tłumaczeniu przez Wydawnictwo Helion.
 * Ten kod jest chroniony prawem autorskim. Nie może być używany do tworzenia materiałów szkoleniowych,
 * kursów, książek, artykułów, itp. W przypadku wątpliwości, skontaktuj się z nami.
 * Nie dajemy gwarancji, że ten kod będzie przydatny do jakiegokolwiek celu.
 * Więcej informacji na temat książki znajdziesz pod adresem  http://www.pragmaticprogrammer.com/titles/tpp20.
***/
/**
  * Z podrozdziału "Szybkość algorytmu". Implementacje
  * popularnych algorytmów sortowania.
  */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "tests.h"

typedef unsigned long ELEMENT;

const int MAX_SIZE=10000000;
const int TEST_SIZE = 1000;

ELEMENT *array = 0;

ELEMENT *tmp_array;

#define SWAP(A,B) { ELEMENT tmp; tmp = A; A = B; B = tmp; }

/*
 * Prototyp generycznej funkcji sortującej
 */

typedef void (*SORT_FUNCTION)(ELEMENT *, int);

/*
 * Sortowanie przez wybór
 */

void selection(ELEMENT *input, int n) {
  int i, j;

  for (i = 0; i < n-1; i++) {
    int     lowest_index = i;
    ELEMENT lowest_value = input[i];

    for (j = i+1; j < n; j++)
      if (input[j] < lowest_value) {
	lowest_value = input[j];
	lowest_index = j;
      }

    SWAP(input[i], input[lowest_index]);
  }
}

/*
 * Sortowanie bąbelkowe
 */

void bubble(ELEMENT *input, int n) {
  ELEMENT tmp;
  int i;

  do {
    
    tmp = input[0];

    for (i = 1; i < n; i++) {
      if (input[i-1] > input[i]) {
	tmp = input[i-1];
	input[i-1] = input[i];
	input[i] = tmp;
      }
    }
  } while (tmp != input[0]);
}


/*
 * Sortowanie przez wstawianie
 */

void insertion(ELEMENT *input, int n) {

  int i, j;
  ELEMENT tmp;

  for (i = 1; i < n; i++) {
    tmp = input[i];
    j = i;

    while ((j > 0) && (input[j-1] > tmp)) {
      input[j] = input[j-1];
      j--;
    }
    input[j] = tmp;
  }
}

/* quicksort
 */

int partition(ELEMENT *input, int first, int last) {
  ELEMENT pivot = input[first];
  int     right = last;
  int     left  = first+1;

  while (1) {

    while (input[right] > pivot) right--;
    while ((left < right) &&
	   input[left] <= pivot) left++;
    if (left >= right)
      break;
    SWAP(input[left], input[right]);
  }

  input[first] = input[right];
  input[right] = pivot;
  return right;
}

void quicksort(ELEMENT *input, int first, int last) {

  if (last > first) {
    int pivot_index = partition(input, first, last);
    quicksort(input, first, pivot_index-1);
    quicksort(input, pivot_index+1, last);
  }
}

void quick(ELEMENT *input, int count) {
  quicksort(input, 0, count-1);
}

/*
 * Sortowanie powłokowe
 */

void shell(ELEMENT *input, int n) {
  int      h = 1;
  ELEMENT  tmp;
  int      i, j;

  do { h = h*3+1; } while (h <= n);

  do {
    h = h/3;

    for (i = h; i < n; i++) {

      tmp = input[i];
      j = i;

      while (input[j-h]>tmp) {
	input[j] = input[j-h];
	j -= h;
	if (j < h)
	  break;
      }
      input[j] = tmp;
    }
  } while (h > 1);
}

/*
 * Proste sortowanie pozycyjne – UWAGA – sortuje wyłącznie liczby bez znaku
 */
void radix(ELEMENT *input, int n) {
  long count[256];

  int shift = 0;
  int i;

  while (shift < 32) {
    memset(count, 0, sizeof(count));

    for (i = 0; i < n; i++) {
      int bits = (input[i] >> shift) & 255;
      count[bits]++;
    }

    for (i = 1; i < 256; i++)
      count[i] += count[i-1];

    for (i = n-1; i >= 0; i--) {
      int bits = (input[i] >> shift) & 255;
      tmp_array[count[bits]] = input[i];
      count[bits]--;
    }

    memcpy(input, tmp_array, n*sizeof(input[0]));

    shift += 8;
  }
}

/*
 * Sprawdzenie, czy tablica jest posortowana. Jeśli nie, wykonaj instrukcję exit(1)
 */

void checkSorted(ELEMENT *array, int size, const char *type, const char *msg) {
  int i;

  for (i = 1; i < size; i++) {
    if (array[i-1] > array[i]) {
      fail("%s: niepowodzenie testu '%s'", type, msg);
      exit(1);
    }
  }
}

/*
 * testSort – uruchamia funkcję sortującą i sprawdza, czy wykonała się prawidłowo
 */

void testSort(SORT_FUNCTION fn, const char *name) {

  int j;

  /* inicjalizacja tablicy losowymi liczbami */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = random();

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "random fill");

  /* inicjalizacja tablicy identycznymi liczbami */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = 0;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "zeroes");

  /* inicjalizacja tablicy liczbami rosnącymi */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = j;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "ascending");

  /* inicjalizacja tablicy liczbami malejącymi */
  for (j = 0; j < TEST_SIZE; j++)
    array[j] = TEST_SIZE - j;

  (*fn)(array, TEST_SIZE);
  checkSorted(array, TEST_SIZE, name, "descending");
}

/*
 * runSort – uruchomienie każdego algorytmu sortowania dla rosnących rozmiarów tablicy
 */

void runSort(SORT_FUNCTION fn, const char *name, 
	     int min, int max, int step, int test) {

  int howBig;

  /* zadbaj o to, by zacząć za każdym razem od tego samego */
  srandom(1);

  if (test) {
    testSort(fn, name);
  }
  else {

    howBig = min;
    
    do {
      int j;
      struct timeval start, end;
      double diff;
      
      /* inicjalizacja tablicy liczbami do posortowania*/
      for (j = 0; j < howBig; j++)
	array[j] = random();
      
      gettimeofday(&start, 0);
      
      (*fn)(array, howBig);
      
      gettimeofday(&end, 0);
      
      diff = (end.tv_sec - start.tv_sec) + 
	(double)(end.tv_usec-start.tv_usec)/1e6;
      
      printf("%-10s %8d  %g\n", name, howBig, diff);
      
      howBig += step;
      
    } while (howBig <= max);
  }
}

/*
 * Wywołanie :
 *
 *    ./sort [-t] [nazwa ...]
 *
 * gdzie 'nazwa' oznacza jeden lub więcej typów sortowania. W przypadku użycia opcji '-t', operacje
 * sortowania są testowane zamiast mierzenia ich czasu wykonania
 *
 */

int main(int argc, char **argv) {

  int testFlag = 0;
  long arraySize = MAX_SIZE;

  argc--; argv++;

  if (argc && (strcmp(*argv, "-t") == 0)) {
    argc--;
    argv++;
    testFlag++;
    arraySize = TEST_SIZE;
  }

  /*
   * Z powodów wydajności wstępnie alokujemy tablicę
   */

  array = (ELEMENT *)malloc(arraySize*sizeof(array[0]));

  if (array == 0) {
    fprintf(stderr, "Brak pamięci na tablicę \n");
    exit(1);
  }

  /*
   * Uruchomienie sortowania
   */
  while (argc--) {

    if (strcmp(*argv, "radix") == 0) {
      tmp_array = (ELEMENT *)malloc(arraySize*sizeof(array[0]));
      runSort(radix, "radix", 1000000, MAX_SIZE, 1000000, testFlag);
      free(tmp_array);
    }
    else if (strcmp(*argv, "quick") == 0)
      runSort(quick, "quick", 1000000, MAX_SIZE, 1000000, testFlag);
    else if (strcmp(*argv, "selection") == 0)
      runSort(selection, "selection", 10000, 100000, 10000, testFlag);
    else if (strcmp(*argv, "shell") == 0)
      runSort(shell, "shell", 10000, MAX_SIZE, 10000, testFlag);
    else if (strcmp(*argv, "insertion") == 0)
      runSort(insertion, "insertion", 10000, 100000, 100000, testFlag);
    else if (strcmp(*argv, "bubble") == 0)
      runSort(bubble, "bubble", 10000, 100000, 100000, testFlag);
    else 
      fprintf(stderr, "Unknown sort type '%s'\n", *argv);

    argv++;
  }

  exit(0);
}
