using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Marketplace.EventSourcing.Logging;

namespace Marketplace.EventSourcing
{
    public abstract class ApplicationService<T> where T : AggregateRoot
    {
        readonly Dictionary<Type, Func<object, Task>> _handlers =
            new Dictionary<Type, Func<object, Task>>();

        readonly IAggregateStore _store;

        protected ApplicationService(IAggregateStore store) => _store = store;

        static ILog Log => LogProvider.GetCurrentClassLogger();

        public Task Handle<TCommand>(TCommand command)
        {
            if (!_handlers.TryGetValue(typeof(TCommand), out var handler))
                throw new InvalidOperationException(
                    $"Nie ma zarejestrowanej procedury obsugi dla polecenia {typeof(TCommand).Name}"
                );

            Log.DebugFormat("Obsuga polecenia: {command}", command.ToString());
            return handler(command);
        }

        protected void CreateWhen<TCommand>(
            Func<TCommand, AggregateId<T>> getAggregateId,
            Func<TCommand, AggregateId<T>, T> creator)
            where TCommand : class
            => When<TCommand>(
                async command =>
                {
                    var aggregateId = getAggregateId(command);

                    if (await _store.Exists(aggregateId))
                        throw new InvalidOperationException(
                            $"Encja z id {aggregateId.ToString()} ju istnieje"
                        );

                    var aggregate = creator(command, aggregateId);

                    await _store.Save(aggregate);
                }
            );

        protected void UpdateWhen<TCommand>(
            Func<TCommand, AggregateId<T>> getAggregateId,
            Action<T, TCommand> updater) where TCommand : class
            => When<TCommand>(
                async command =>
                {
                    var aggregateId = getAggregateId(command);
                    var aggregate = await _store.Load(aggregateId);

                    if (aggregate == null)
                        throw new InvalidOperationException(
                            $"Nie mona znale encji z id {aggregateId.ToString()}"
                        );

                    updater(aggregate, command);
                    await _store.Save(aggregate);
                }
            );

        protected void UpdateWhen<TCommand>(
            Func<TCommand, AggregateId<T>> getAggregateId,
            Func<T, TCommand, Task> updater) where TCommand : class
            => When<TCommand>(
                async command =>
                {
                    var aggregateId = getAggregateId(command);
                    var aggregate = await _store.Load(aggregateId);

                    if (aggregate == null)
                        throw new InvalidOperationException(
                            $"Nie mona znale encji z id {aggregateId.ToString()}"
                        );

                    await updater(aggregate, command);
                    await _store.Save(aggregate);
                }
            );

        void When<TCommand>(Func<TCommand, Task> handler)
            where TCommand : class
            => _handlers.Add(typeof(TCommand), c => handler((TCommand) c));
    }
}