package com.wordz.adapters.db;

import com.github.database.rider.core.api.configuration.DBUnit;
import com.github.database.rider.core.api.configuration.Orthography;
import com.github.database.rider.core.api.connection.ConnectionHolder;
import com.github.database.rider.core.api.dataset.DataSet;
import com.github.database.rider.core.api.dataset.ExpectedDataSet;
import com.github.database.rider.junit5.api.DBRider;
import com.wordz.domain.Game;
import com.wordz.domain.GameRepository;
import com.wordz.domain.Player;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import javax.sql.DataSource;
import java.util.Optional;

import static org.assertj.core.api.Assertions.assertThat;

@DBRider
@DBUnit(caseSensitiveTableNames = true,
        caseInsensitiveStrategy = Orthography.LOWERCASE)
public class GameRepositoryPostgresTest {

    private DataSource dataSource;

    @SuppressWarnings("unused") // Używane przez framework DBRider
    private final ConnectionHolder connectionHolder = () -> dataSource.getConnection();

    /**
     * Ta metoda konfiguruje połączenie z testową bazą danych.
     * Ponieważ nie tworzy przy tej okazji żadnych danych testowych,
     * moglibyśmy rozważyć użycie adnotacji @BeforeAll, żeby przeprowadzić
     * tę samą operację jednokrotnie, a nie dla każdego przypadku testowego.
     */
    @BeforeEach
    void setupConnection() {
        this.dataSource = new PostgresTestDataSource();
    }

    /**
     * Zapobieganie zapisywaniu powielonych danych:
     * Jednym z problemów w kroku przygotowania testu bazodanowego jest to,
     * że chcielibyśmy wstawić konkretny rekord do bazy danych, aby móc napisać krok asercji.
     * Najprostszym rozwiązaniem byłoby zapisywanie tego samego rekordu za każdym razem.
     *
     * Baza danych będzie go przechowywać do czasu wykasowania go lub usunięcia całej tabeli.
     * Powoduje to jednak problemy z izolacją testów, która jest jedną z zasad FIRST.
     * Jeżeli uruchomimy ten sam test ponownie, drugie uruchomienie zakończy się niepowodzeniem.
     * Instrukcja INSERT zgłosi błąd, ponieważ rekord już istnieje.
     * Został utworzony w wyniku uruchomienia poprzedniego testu.
     *
     * Istnieją dwa dobre podejścia do rozwiązania tego problemu:
     * - usunięcie wszystkich danych z tabeli (lub wykasowanie rekordu) przed każdym uruchomieniem testu;
     * - utworzenie rekordów z niepowtarzalnymi wartościami.
     *
     * Usuwanie wszystkich danych z tabeli działa dobrze w przypadku programistycznych środowisk testowych.
     * Jeżeli jesteśmy w stanie odtworzyć początkowy stan tabeli przed każdym testem,
     * możemy użyć tego podejścia. To właśnie rozwiązanie wybraliśmy dla tego testu.
     * Oczywiście nie nadaje się ono do testowania na produkcji. Podobne problemy generuje kasowanie rekordów.
     *
     * Tworzenie rekordów z niepowtarzalnymi wartościami to inne rozsądne podejście.
     * Generatora losowych wartości można użyć do tworzenia sztucznych liczb i ciągów znaków.
     * Możemy na przykład utworzyć nazwisko "surname-03defa21". Chodzi o to, że przy uruchomieniu testu
     * za każdym razem nastąpi zapisanie nowej wartości, dzięki czemu można będzie uniknąć konfliktów.
     * Okresowo należy jednak kasować wszystkie rekordy z bazy.
     *
     * Wykasowanie wszystkich danych z tabeli po uruchomieniu testu wydaje się kuszącym rozwiązaniem.
     * Zazwyczaj zwalniamy zasoby po uruchomieniu testów. W przypadku testowania bazy danych takie zwolnienie
     * często nigdy nie dochodziłoby do skutku. Jeżeli wydarzy się jakiś krytyczny błąd podczas testu,
     * zwolnienie zasobów zostanie pominięte. Dlatego lepiej jest uruchamiać kod czyszczący przed rozpoczęciem testu.
     *
     * Kasowanie wszystkich danych z tabel za pomocą frameworku DBRider:
     * Wartość true atrybutu cleanBefore adnotacji @DataSet informuje framework DBRider o konieczności
     * wykasowania wszystkich rekordów w tabeli na początku testu.
     */
    @Test
    @DataSet(value = "adapters/data/emptyGame.json",
            cleanBefore = true)
    @ExpectedDataSet(value = "/adapters/data/createGame.json")
    public void storesNewGame() {
        var player = new Player("gracz1");
        var game = new Game(player, "BONUS", 0, false);

        GameRepository games = new GameRepositoryPostgres(dataSource);
        games.create(game);
    }

    /**
     * Określanie początkowych i oczekiwanych wartości w bazie danych:
     * Pewnym wyzwaniem przy pisaniu testów bazodanowych jest zapewnienie czytelności
     * danych początkowych w kroku przygotowania oraz oczekiwanych danych w asercji.
     *
     * Istnieje kilka użytecznych podejść w tym zakresie:
     * - użycie wzorca projektowego typu kaskadowy budowniczy (ang. fluid builder);
     * - surowe instrukcje SQL w samym teście;
     * - pliki z danymi.
     *
     * Problemy z czytelnością pojawiają się, ponieważ baza danych jest czymś zewnętrznym
     * względem kodu domeny w Javie. Bazy danych wykorzystują zazwyczaj własne języki, takie jak SQL.
     * Aby połączyć się z bazą danych, konieczne jest zastosowanie jakiegoś rodzaju biblioteki (np. JDBC).
     * W związku z tym nie możemy użyć czystego kodu Javy tak jak w modelu domeny.
     *
     * Wzorzec projektowy kaskadowego budowniczego znajdziesz tutaj:
     * <a href="https://en.wikipedia.org/wiki/Fluent_interface#Java">Fluent Interface na Wikipedii</a>.
     * Dzięki temu wzorcowi możemy zbudować skomplikowaną strukturę danych krok po kroku.
     * Przewaga jest taka, że używamy Javy do opisania procesu budowania, którego można użyć
     * w kroku przygotowania. Możemy potem łatwo odczytać wszystkie utworzone dane.
     * Bliskie umiejscowienie danych użytych w kodzie testowym jest istotne.
     *
     * W podobny sposób użycie surowych instrukcji SQL w kroku przygotowania zapewnia widoczność.
     * Używając metody z biblioteki JDBC, możemy napisać mniej więcej taką linię kodu:
     * executeInsert( "INSERT INTO users VALUES (1, 'Alan')" );.
     * To również utrzymuje dane bliżej kodu testowego, który od nich zależy.
     *
     * Końcowe podejście preferuje zachowanie danych w ich naturalnym formacie względem
     * utrzymywania danych blisko testu. W tym przypadku używamy plików z danymi umieszczonych
     * w katalogu test/resources projektu w Javie. Kod testowy odnosi się do tych plików
     * po nazwie, co oznacza, że musimy otworzyć plik, aby dowiedzieć się, jakich danych używamy.
     * Dane są jednak zapisane w jakimś popularnym formacie, takim jak CSV lub JSON.
     * Takie dane można łatwiej dostarczyć i obsługiwać narzędziami zewnętrznymi.
     * Musimy jednak zrezygnować z umiejscowienia danych blisko kodu.
     *
     * Podejście obrane w tych testach to pliki danych. Jest ono dobrze wspierane przez framework DBRider.
     * Tworzymy plik dla początkowej konfiguracji bazy danych w kroku przygotowania oraz dla oczekiwanych danych.
     * Framework zapewnia dwie adnotacje, które pozwalają podłączyć te pliki z testem. Adnotacja @DataSet()
     * odnosi się do początkowych danych, a @ExpectedDataSet() wskazuje na dane do asercji.
     *
     * Zwróć uwagę, że adnotacje wykonują części testów. Adnotacja @DataSet() wstawi dane do bazy danych.
     * Adnotacja @ExpectedDataSet uruchomi kod asercji. W związku z tym test wygląda na pierwszy rzut oka
     * na niedokończony. Kroki przygotowania i asercji są realizowane głównie przez kod frameworku DBRider
     * aktywowany przez adnotacje.
     */
    @Test
    @DataSet(value = "adapters/data/createGame.json",
            cleanBefore = true)
    public void fetchesGame() {
        GameRepository games = new GameRepositoryPostgres(dataSource);

        var player = new Player("gracz1");
        Optional<Game> game = games.fetchForPlayer(player);

        assertThat(game.isPresent()).isTrue();
        var actual = game.get();
        assertThat(actual.getPlayer()).isEqualTo(player);
        assertThat(actual.getWord()).isEqualTo("BONUS");
        assertThat(actual.getAttemptNumber()).isZero();
        assertThat(actual.isGameOver()).isFalse();
    }

    @Test
    @DataSet(value = "adapters/data/emptyGame.json",
            cleanBefore = true)
    public void reportsGameNotFoundForPlayer() {
        GameRepository games = new GameRepositoryPostgres(dataSource);

        var player = new Player("gracz1");
        Optional<Game> game = games.fetchForPlayer(player);

        assertThat(game.isEmpty()).isTrue();
    }

    @Test
    @DataSet(value = "adapters/data/createGame.json",
            cleanBefore = true)
    @ExpectedDataSet(value = "/adapters/data/updatedGame.json")
    public void updatesGame() {
        var player = new Player("gracz1");
        var game = new Game(player, "BONUS", 0, false);

        game.attempt("AAAAA");
        GameRepository games = new GameRepositoryPostgres(dataSource);

        games.update(game);
    }
}
