'use strict';

const Sequelize = require("sequelize");
const jsyaml    = require('js-yaml');
const fs        = require('fs');
const util      = require('util');

const log   = require('debug')('users:model-users');
const error = require('debug')('users:error');

var SQUser;
var sequlz;

exports.connectDB = function() {
    
    if (SQUser) return SQUser.sync();
    
    return new Promise((resolve, reject) => {
        fs.readFile(process.env.SEQUELIZE_CONNECT, 'utf8', (err, data) => {
            if (err) reject(err);
            else resolve(data);
        });
    })
    .then(yamltext => {
        return jsyaml.safeLoad(yamltext, 'utf8');
    })
    .then(params => {
        log('Parametry Sequelize '+ util.inspect(params));
        
        if (!sequlz) sequlz = new Sequelize(params.dbname, params.username, params.password, params.params);
        
        // Te pola pochodzą głównie ze schematu Portable Contacts z narzędzia Passport
        // Zobacz http://www.passportjs.org/docs/profile
        //
        // Pola emails i photos w schemacie Portable Contacts to tablice. Wymagają utworzenia 
        // dodatkowych tabel.
        //
        // Pole "id" ze schematu Portable Contacts to pole "username" w tym kodzie
        if (!SQUser) SQUser = sequlz.define('User', {
            username: { type: Sequelize.STRING, unique: true },
            password: Sequelize.STRING,
            provider: Sequelize.STRING,
            familyName: Sequelize.STRING,
            givenName: Sequelize.STRING,
            middleName: Sequelize.STRING,
            emails: Sequelize.STRING(2048),
            photos: Sequelize.STRING(2048)
        });
        return SQUser.sync();
    });
};

exports.create = function(username, password, provider, familyName, givenName, middleName, emails, photos) {
    return exports.connectDB().then(SQUser => {
        return SQUser.create({
            username: username,
            password: password,
            provider: provider,
            familyName: familyName,
            givenName: givenName,
            middleName: middleName,
            emails: JSON.stringify(emails),
            photos: JSON.stringify(photos)
        });
    });
};

exports.update = function(username, password, provider, familyName, givenName, middleName, emails, photos) {
    return exports.find(username).then(user => {
        return user ? user.updateAttributes({
            password: password,
            provider: provider,
            familyName: familyName,
            givenName: givenName,
            middleName: middleName,
            emails: JSON.stringify(emails),
            photos: JSON.stringify(photos)
        }) : undefined;
    });
};

exports.destroy = function(username) {
    return exports.connectDB().then(SQUser => {
        return SQUser.find({ where: { username: username } })
    })
    .then(user => {
        if (!user) throw new Error('Nie znaleziono żądanego konta '+ username +' do usunięcia');
        user.destroy();
        return;
    });
};

exports.find = function(username) {
    log('find  '+ username);
    return exports.connectDB().then(SQUser => {
        return SQUser.find({ where: { username: username } })
    })
    .then(user => user ? exports.sanitizedUser(user) : undefined);
};

exports.userPasswordCheck = function(username, password) {
    log('userPasswordCheck query= '+ username +' '+ password);
    return exports.connectDB().then(SQUser => {
        return SQUser.find({ where: { username: username } })
    })
    .then(user => {
        log('userPasswordCheck query= '+ username +' '+ password
            +' user= '+ (user && user != null ? user.username : "") +' '+ (user && user != null ? user.password : ""));
        if (!user) {
            return { check: false, username: username, message: "Nie można znaleźć użytkownika" };
        } else if (user.username === username && user.password === password) {
            return { check: true, username: user.username };
        } else {
            return { check: false, username: username, message: "Błędne hasło" };
        }
    });
};

exports.findOrCreate = function(profile) {
    return exports.find(profile.id).then(user => {
        if (user) return user;
        log('findOrCreate tworzy => '+ util.inspect(profile));
        return exports.create(profile.id, profile.password, profile.provider,
                       profile.familyName, profile.givenName, profile.middleName,
                       profile.emails, profile.photos);
    });
};

exports.listUsers = function() {
    return exports.connectDB()
    .then(SQUser => SQUser.findAll({}) )
    .then(userlist => userlist.map(user => exports.sanitizedUser(user)))
    .catch(err => console.error(err));
};

exports.sanitizedUser = function(user) {
    log(util.inspect(user));
    if (!user) throw new Error('Brak użytkownika do poprawienia')
    return {
        id: user.username,
        username: user.username,
        provider: user.provider,
        familyName: user.familyName,
        givenName: user.givenName,
        middleName: user.middleName,
        emails: user.emails,
        photos: user.photos
    };
};
