// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.TextBoxWatermarkExtender Wrapper
    /// </summary>
    /// <TestComponent Name="TextboxWatermark">
    ///   <ToolkitType>AjaxControlToolkit.TextBoxWatermarkExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.TextboxWatermark.TextboxWatermark_Common</CommonTestSuite>
    /// </TestComponent>
    public class TextboxWatermarkBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// WatermarkText
        /// </summary>
        public BehaviorProperty<string> WatermarkText
        {
            get { return _watermarkText; }
        }
        private BehaviorProperty<string> _watermarkText;

        /// <summary>
        /// WatermarkCssClass
        /// </summary>
        public BehaviorProperty<string> WatermarkCssClass
        {
            get { return _watermarkCssClass; }
        }
        private BehaviorProperty<string> _watermarkCssClass;

        /// <summary>
        /// Text
        /// </summary>
        public BehaviorProperty<string> Text
        {
            get { return _text; }
        }
        private BehaviorProperty<string> _text;

        /// <summary>
        /// Element's text
        /// </summary>
        public BehaviorProperty<string> ElementText
        {
            get { return _elementText; }
        }
        private BehaviorProperty<string> _elementText;

        /// <summary>
        /// Element's CSS class
        /// </summary>
        public BehaviorProperty<string> ElementCssClass
        {
            get { return _elementCssClass; }
        }
        private BehaviorProperty<string> _elementCssClass;

        /// <summary>
        /// Set Text property
        /// </summary>
        /// <param name="text">value</param>
        public void SetText(string text)
        {
            Text.Value = text;
            Invalidate();
        }

        /// <summary>
        /// Set WatermarkText property
        /// </summary>
        /// <param name="text">value</param>
        public void SetWatermarkText(string text)
        {
            WatermarkText.Value = text;
            Invalidate();
        }

        /// <summary>
        /// Set WatermarkCssClass property
        /// </summary>
        /// <param name="cssClass">value</param>
        public void SetWatermarkCssClass(string cssClass)
        {
            WatermarkCssClass.Value = cssClass;
            Invalidate();
        }

        /// <summary>
        /// Set Element.value
        /// </summary>
        /// <param name="text">value</param>
        public void SetElementText(string text)
        {
            Element.SetText(text, false);
            Invalidate();
        }

        /// <summary>
        /// Focus the element
        /// </summary>
        public void Focus()
        {
            Element.Focus();
            Invalidate();
        }

        /// <summary>
        /// Blur the element
        /// </summary>
        public void Blur()
        {
            Element.Blur();
            Invalidate();
        }

        /// <summary>
        /// AjaxControlToolkit.TextBoxWatermarkExtender Wrapper
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public TextboxWatermarkBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _watermarkText = BehaviorProperty<string>.CreateProperty(this, "WatermarkText");
            _watermarkCssClass = BehaviorProperty<string>.CreateProperty(this, "WatermarkCssClass");
            _text = BehaviorProperty<string>.CreateProperty(this, "Text");
            _elementText = BehaviorProperty<string>.CreateField(this, "value", "{0}._element");
            _elementCssClass = BehaviorProperty<string>.CreateField(this, "className", "{0}._element");
        }

        public void AssertWatermarked()
        {
            Require(ElementCssClass, Text, WatermarkText, WatermarkCssClass);
            Assert.StringIsNullOrEmpty(Text.Value);
            Assert.AreEqual(WatermarkText.Value, ElementText.Value);  // TODO: Resolve occasional race condition
            Assert.AreEqual(WatermarkCssClass.Value, ElementCssClass.Value);
        }

        public void AssertFull(string text)
        {
            Require(ElementCssClass, Text, WatermarkText, WatermarkCssClass);
            Assert.AreEqual(text, Text.Value);
            Assert.AreEqual(text, ElementText.Value);  // TODO: Resolve occasional race condition
            Assert.AreNotEqual(WatermarkCssClass.Value, ElementCssClass.Value);
        }

        public void AssertEmpty()
        {
            Require(ElementCssClass, Text, WatermarkText, WatermarkCssClass);
            Assert.StringIsNullOrEmpty(Text.Value);
            Assert.StringIsNullOrEmpty(ElementText.Value);  // TODO: Resolve occasional race condition
            Assert.AreNotEqual(WatermarkCssClass.Value, ElementCssClass.Value);
        }
    }
}
