// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.RatingExtender Model
    /// </summary>
    /// <TestComponent Name="Rating">
    ///   <ToolkitType>AjaxControlToolkit.RatingExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.Rating.Rating_Common</CommonTestSuite>
    /// </TestComponent>
    public class RatingBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// AutoPostBack
        /// </summary>
        public BehaviorProperty<bool> AutoPostBack
        {
            get { return _autoPostBack; }
        }
        private BehaviorProperty<bool> _autoPostBack;

        /// <summary>
        /// Rating
        /// </summary>
        public BehaviorProperty<int> Rating
        {
            get { return _rating; }
        }
        private BehaviorProperty<int> _rating;

        /// <summary>
        /// CallbackID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> CallbackControl
        {
            get { return _callbackControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _callbackControl;

        /// <summary>
        /// Tag
        /// </summary>
        public BehaviorProperty<string> Tag
        {
            get { return _tag; }
        }
        private BehaviorProperty<string> _tag;

        /// <summary>
        /// RatingDirection
        /// </summary>
        public BehaviorProperty<RatingDirection> RatingDirection
        {
            get { return _ratingDirection; }
        }
        private BehaviorProperty<RatingDirection> _ratingDirection;

        /// <summary>
        /// MaxRating
        /// </summary>
        public BehaviorProperty<int> MaxRating
        {
            get { return _maxRating; }
        }
        private BehaviorProperty<int> _maxRating;

        /// <summary>
        /// StarCssClass
        /// </summary>
        public BehaviorProperty<string> StarCssClass
        {
            get { return _starCssClass; }
        }
        private BehaviorProperty<string> _starCssClass;

        /// <summary>
        /// ReadOnly
        /// </summary>
        public BehaviorProperty<bool> ReadOnly
        {
            get { return _readOnly; }
        }
        private BehaviorProperty<bool> _readOnly;

        /// <summary>
        /// FilledStarCssClass
        /// </summary>
        public BehaviorProperty<string> FilledStarCssClass
        {
            get { return _filledStarCssClass; }
        }
        private BehaviorProperty<string> _filledStarCssClass;

        /// <summary>
        /// EmptyStarCssClass
        /// </summary>
        public BehaviorProperty<string> EmptyStarCssClass
        {
            get { return _emptyStarCssClass; }
        }
        private BehaviorProperty<string> _emptyStarCssClass;

        /// <summary>
        /// WaitingStarCssClass
        /// </summary>
        public BehaviorProperty<string> WaitingStarCssClass
        {
            get { return _waitingStarCssClass; }
        }
        private BehaviorProperty<string> _waitingStarCssClass;

        /// <summary>
        /// Rating stars collection
        /// </summary>
        public HtmlElementCollection GetStars()
        {
            if (this._stars == null)
            {
                this._stars = this.Element.ChildElements.Find("a", 0).ChildElements;
                Assert.IsNotNull(this._stars);
                Assert.AreEqual(this.MaxRating.Value, this._stars.Count);
            }
            return this._stars;
        }
        private HtmlElementCollection _stars;

        /// <summary>
        /// Rated
        /// </summary>
        public BehaviorEvent Rated
        {
            get { return _rated; }
        }
        private BehaviorEvent _rated;

        /// <summary>
        /// AjaxControlToolkit.RatingExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public RatingBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _autoPostBack = BehaviorProperty<bool>.CreateProperty(this, "AutoPostBack");
            _rating = BehaviorProperty<int>.CreateProperty(this, "Rating", ReadStrategy.Volatile, WriteStrategy.Immediate);
            _callbackControl = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "CallbackID");
            _tag = BehaviorProperty<string>.CreateProperty(this, "Tag");
            _ratingDirection = BehaviorProperty<AjaxControlToolkit.RatingDirection>.CreateProperty(this, "RatingDirection");
            _maxRating = BehaviorProperty<int>.CreateProperty(this, "MaxRating");
            _starCssClass = BehaviorProperty<string>.CreateProperty(this, "StarCssClass");
            _readOnly = BehaviorProperty<bool>.CreateProperty(this, "ReadOnly");
            _filledStarCssClass = BehaviorProperty<string>.CreateProperty(this, "FilledStarCssClass");
            _emptyStarCssClass = BehaviorProperty<string>.CreateProperty(this, "EmptyStarCssClass");
            _waitingStarCssClass = BehaviorProperty<string>.CreateProperty(this, "WaitingStarCssClass");
            _rated = BehaviorEvent<object>.CreateEvent(this, "Rated");

        }

        /// <summary>
        /// Click on the specified rating. Depending on the direction of the 
        /// rating control the right rating will be selected. For example if Rating is 
        /// 1 and the direction is righttoleft then the right-most rating will be selected.
        /// </summary>
        public void ClickRating(int rating, bool waitForPostback)
        {
            if (rating > 0 && rating <= this.MaxRating.Value)
            {
                this.Rated.Register(); 
                HtmlElement starToClick = null;
                if (this.RatingDirection.Value == AjaxControlToolkit.RatingDirection.LeftToRightTopToBottom)
                {
                    starToClick = this.GetStars()[rating - 1];
                }
                else
                {
                    starToClick = this.GetStars()[(this.MaxRating.Value - rating)];
                }
                starToClick.DispatchEvent(new HtmlMouseEvent("mouseover"));
                starToClick.Click(waitForPostback, true);
                this.Rated.Wait(5, false);
                Assert.AreEqual(rating, this.Rating.Value);
                // this line may fail since the Rated event is raised before 
                // the styles are updated
                this.AssertStarsCss();
            }
        }

        public void AssertStarsCss()
        {
            int currentRating = this.Rating.Value;
            string script = 
                    "(function () {" +
                    "var stars = " + this.BehaviorReferenceExpression + ".get_Stars();" +
                    "var starStyles = {};" +
                    "for (var i = 0; i < " + this.MaxRating.Value + "; ++i) {" +
                    "starStyles[i] = stars[i].className;" +
                    "}" +
                    "return starStyles;})()";
            IDictionary<string, object> starCssStyles = Common.ConvertJsonValue<IDictionary<string, object>>(this.Page.ExecuteScript(script));
            if (this.RatingDirection.Value == AjaxControlToolkit.RatingDirection.LeftToRightTopToBottom)
            {
                for (int i = 0; i < currentRating; ++i)
                {
                    this.AssertStyles(starCssStyles[i.ToString()].ToString(), this.FilledStarCssClass.Value, this.StarCssClass.Value);
                }

                for (int i = currentRating; i < this.MaxRating.Value; ++i)
                {
                    this.AssertStyles(starCssStyles[i.ToString()].ToString(), this.EmptyStarCssClass.Value, this.StarCssClass.Value);
                }
            }
            else
            {
                for (int i = this.MaxRating.Value - 1; i >= (this.MaxRating.Value - currentRating); --i)
                {
                    this.AssertStyles(starCssStyles[i.ToString()].ToString(), this.FilledStarCssClass.Value, this.StarCssClass.Value);
                }

                for (int i = 0; i < (this.MaxRating.Value - currentRating); ++i)
                {
                    this.AssertStyles(starCssStyles[i.ToString()].ToString(), this.EmptyStarCssClass.Value, this.StarCssClass.Value);
                }
            }
        }

        private void AssertStyles(string currentStyles, params string[] expectedStyles)
        {
            foreach (string expectedStyle in expectedStyles)
            {
                Assert.StringContains(currentStyles, expectedStyle);
            }
        }
    }
}