// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.PagingBulletedListExtender Model
    /// </summary>
    /// <TestComponent Name="PagingBulletedList">
    ///   <ToolkitType>AjaxControlToolkit.PagingBulletedListExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.PagingBulletedList.PagingBulletedList_Common</CommonTestSuite>
    /// </TestComponent>
    public class PagingBulletedListBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// IndexSize
        /// </summary>
        public BehaviorProperty<int> IndexSize
        {
            get { return _indexSize; }
        }
        private BehaviorProperty<int> _indexSize;

        /// <summary>
        /// Height
        /// </summary>
        public BehaviorProperty<int?> Height
        {
            get { return _height; }
        }
        private BehaviorProperty<int?> _height;

        /// <summary>
        /// Separator
        /// </summary>
        public BehaviorProperty<string> Separator
        {
            get { return _separator; }
        }
        private BehaviorProperty<string> _separator;

        /// <summary>
        /// MaximumItemsPerPage
        /// </summary>
        public BehaviorProperty<int?> MaximumItemsPerPage
        {
            get { return _maximumItemsPerPage; }
        }
        private BehaviorProperty<int?> _maximumItemsPerPage;

        /// <summary>
        /// SortOnClient
        /// </summary>
        public BehaviorProperty<bool> SortOnClient
        {
            get { return _sortOnClient; }
        }
        private BehaviorProperty<bool> _sortOnClient;

        /// <summary>
        /// SelectedIndexCssClass
        /// </summary>
        public BehaviorProperty<string> SelectedIndexCssClass
        {
            get { return _selectedIndexCssClass; }
        }
        private BehaviorProperty<string> _selectedIndexCssClass;

        /// <summary>
        /// IndexCssClass
        /// </summary>
        public BehaviorProperty<string> IndexCssClass
        {
            get { return _indexCssClass; }
        }
        private BehaviorProperty<string> _indexCssClass;

        /// <summary>
        /// SelectedIndex
        /// </summary>
        public BehaviorProperty<int> SelectedIndex
        {
            get { return _selectedIndex; }
        }
        private BehaviorProperty<int> _selectedIndex;

        /// <summary>
        /// Tab indices seperated by a comma
        /// </summary>
        public BehaviorProperty<string> Indices
        {
            get { return _indices; }
        }
        private BehaviorProperty<string> _indices;

        /// <summary>
        /// Current tab values seperated by a comma
        /// </summary>
        public BehaviorProperty<string> CurrentValues
        {
            get { return _currentValues; }
        }
        public BehaviorProperty<string> _currentValues;

        /// <summary>
        /// IndexChanged
        /// </summary>
        public BehaviorEvent IndexChanged
        {
            get { return _indexChanged; }
        }
        private BehaviorEvent _indexChanged;

        /// <summary>
        /// AjaxControlToolkit.PagingBulletedListExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public PagingBulletedListBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _indexSize = BehaviorProperty<int>.CreateProperty(this, "IndexSize");
            _height = BehaviorProperty<int?>.CreateProperty(this, "Height");
            _separator = BehaviorProperty<string>.CreateProperty(this, "Separator");
            _maximumItemsPerPage = BehaviorProperty<int?>.CreateProperty(this, "MaxItemPerPage");
            _sortOnClient = BehaviorProperty<bool>.CreateProperty(this, "ClientSort");
            _selectedIndexCssClass = BehaviorProperty<string>.CreateProperty(this, "SelectIndexCssClass");
            _indexCssClass = BehaviorProperty<string>.CreateProperty(this, "UnselectIndexCssClass");
            _selectedIndex = BehaviorProperty<int>.CreateField(this, "_indexSelected");
            _indices = BehaviorProperty<string>.CreateCustomProperty(this, null, "(function() {{ " +
                    "var indices = Array.clone({0}.get_tabIndex()); " +
                    "if (!indices || indices.length == 0) {{ return null; }} " +
                    "for (var i = 0; i < indices.length; i++) {{ indices[i] = indices[i].innerHTML; }} " +
                    "return indices.join(','); " +
                "}})()", "throw 'Indices property is read-only!'");
            _currentValues = BehaviorProperty<string>.CreateCustomProperty(this, null, "(function() {{ " +
                    "var b = {0}; " +
                    "var values = Array.clone(b._tabValue[b._indexSelected]); " +
                    "if (!values || values.length == 0) {{ return null; }} " +
                    "for (var i = 0; i < values.length; i++) {{ " +
                        "values[i] = (values[i].firstChild) && (values[i].firstChild.innerHTML) ? " +
                            "values[i].firstChild.innerHTML : " +
	                        "values[i].innerHTML; " +
                    "}} " +
                    "return values.join(','); " +
                "}})()", "throw 'CurrentValues property is read-only!'");
            _indexChanged = BehaviorEvent<object>.CreateEvent(this, "indexChanged");
        }

        /// <summary>
        /// Select a given index
        /// </summary>
        /// <param name="index">Position of the index to select</param>
        public void SelectIndex(int index)
        {
            _indexChanged.Register();
            Assert.IsTrue((bool) Page.ExecuteScript(BehaviorReferenceExpression + "._selectIndex(" + index + "), true"),
                "Could not find an index at position \"{0}\" to select!", index);
            _indexChanged.Wait(3);
        }

        /// <summary>
        /// Verify the list of current indices
        /// </summary>
        /// <param name="indices">Indices</param>
        public void AssertIndices(params string[] indices)
        {
            AssertAreEqual((_indices.Value != null ? _indices.Value.Trim().Split(',') : new string[] { }), indices);
        }

        /// <summary>
        /// Verify the list of current values
        /// </summary>
        /// <param name="values">Values</param>
        public void AssertCurrentValues(params string[] values)
        {
            AssertAreEqual((_currentValues.Value != null ? _currentValues.Value.Trim().Split(',') : new string[] { }), values);
        }

        /// <summary>
        /// Verify the equality of the arrays
        /// </summary>
        /// <typeparam name="T">Type of the arrays</typeparam>
        /// <param name="expected">Expected values</param>
        /// <param name="values">Actual values</param>
        private static void AssertAreEqual<T>(T[] expected, T[] values)
        {
            string message = string.Format("Array {0} not equal to expected array {1}!", ToString(values), ToString(expected));
            if (expected == null)
            {
                Assert.IsNull(values, message);
            }
            else
            {
                Assert.IsNotNull(values, message);
            }

            Assert.AreEqual(expected.Length, values.Length, message);
            for (int i = 0; i < values.Length; i++)
            {
                Assert.AreEqual<T>(expected[i], values[i], message);
            }
        }

        /// <summary>
        /// Create a string representation of the arrray
        /// </summary>
        /// <typeparam name="T">Type of the array</typeparam>
        /// <param name="array">Array</param>
        /// <returns>String representation of the array</returns>
        private static string ToString<T>(T[] array)
        {
            return (array != null) ?
                string.Format("[{0}]", string.Join(", ", Array.ConvertAll<T, string>(array, delegate(T value)
                    { return value == null ? "null" : string.Format("<{0}>", value); }))) :
                "null";
        }
    }
}