// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.MutuallyExclusiveCheckBoxExtender Model
    /// </summary>
    /// <TestComponent Name="MutuallyExclusiveCheckBox">
    ///   <ToolkitType>AjaxControlToolkit.MutuallyExclusiveCheckBoxExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.MutuallyExclusiveCheckBox.MutuallyExclusiveCheckBox_Common</CommonTestSuite>
    /// </TestComponent>
    public class MutuallyExclusiveCheckBoxBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// Key
        /// </summary>
        public BehaviorProperty<string> Key
        {
            get { return _key; }
        }
        private BehaviorProperty<string> _key;

        /// <summary>
        /// AjaxControlToolkit.MutuallyExclusiveCheckBoxExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public MutuallyExclusiveCheckBoxBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _key = BehaviorProperty<string>.CreateProperty(this, "Key");
        }

        public void SelectCheckBox()
        {
            this.Element.Click(false);
        }

        public bool IsChecked
        {
            get
            {
                return (this.Element.GetAttributes().Checked != null);
            }
        }

        // validate exclusiveness
        public void AssertAtMostOneChecked()
        {
            Assert.IsTrue(this.GetCheckedCount() <= 1);
        }

        // make sure that no checkbox in this set has been checked
        public void AssertNoneChecked()
        {
            Assert.AreEqual(0, this.GetCheckedCount());
        }

        // make sure that one checkbox in this set has been checked
        public void AssertOneChecked()
        {
            Assert.AreEqual(1, this.GetCheckedCount());
        }

        private int GetCheckedCount()
        {
            int count = 0;
            foreach (MutuallyExclusiveCheckBoxBehavior behavior in this.MutuallyExclusiveCheckBoxSet)
            {
                if (behavior.IsChecked)
                {
                    ++count;
                }
            }
            return count;
        }

        // get all other checkboxes
        public IList<MutuallyExclusiveCheckBoxBehavior> MutuallyExclusiveCheckBoxSet
        {
            get
            {
                if (this._mutuallyExclusiveCheckBoxSet == null)
                {
                    this._mutuallyExclusiveCheckBoxSet = new List<MutuallyExclusiveCheckBoxBehavior>();
                    foreach (Behavior behavior in this.Page.Behaviors)
                    {
                        MutuallyExclusiveCheckBoxBehavior checkBox = behavior as MutuallyExclusiveCheckBoxBehavior;
                        if (checkBox != null)
                        {
                            _mutuallyExclusiveCheckBoxSet.Add(checkBox);
                            checkBox.Key.Require();
                        }
                    }
                    this._mutuallyExclusiveCheckBoxSet.RemoveAll(
                        delegate(MutuallyExclusiveCheckBoxBehavior a)
                        {
                            return (String.Compare(a.Key.Value, this.Key.Value) != 0);
                        });
                }
                return this._mutuallyExclusiveCheckBoxSet;
            }
        }
        private List<MutuallyExclusiveCheckBoxBehavior> _mutuallyExclusiveCheckBoxSet;
    }
}