// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.SlideShowExtender Wrapper
    /// </summary>
    /// <TestComponent Name="SlideShow">
    ///   <ToolkitType>AjaxControlToolkit.SlideShowExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.SlideShow.SlideShow_Common</CommonTestSuite>
    /// </TestComponent>
    public class SlideShowBehavior : Behavior<HtmlImageElement>
    {
        /// <summary>
        /// SlideShowServiceMethod
        /// </summary>
        public BehaviorProperty<string> SlideShowServiceMethod
        {
            get { return _slideShowServiceMethod; }
        }
        private BehaviorProperty<string> _slideShowServiceMethod;

        /// <summary>
        /// SlideShowServicePath
        /// </summary>
        public BehaviorProperty<string> SlideShowServicePath
        {
            get { return _slideShowServicePath; }
        }
        private BehaviorProperty<string> _slideShowServicePath;

        /// <summary>
        /// ContextKey
        /// </summary>
        public BehaviorProperty<string> ContextKey
        {
            get { return _contextKey; }
        }
        private BehaviorProperty<string> _contextKey;

        /// <summary>
        /// UseContextKey
        /// </summary>
        public BehaviorProperty<bool> UseContextKey
        {
            get { return _useContextKey; }
        }
        private BehaviorProperty<bool> _useContextKey;

        /// <summary>
        /// NextButton
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> NextButton
        {
            get { return _nextButton; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _nextButton;

        /// <summary>
        /// PlayButton
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> PlayButton
        {
            get { return _playButton; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _playButton;

        /// <summary>
        /// PlayButtonText
        /// </summary>
        public BehaviorProperty<string> PlayButtonText
        {
            get { return _playButtonText; }
        }
        private BehaviorProperty<string> _playButtonText;

        /// <summary>
        /// StopButtonText
        /// </summary>
        public BehaviorProperty<string> StopButtonText
        {
            get { return _stopButtonText; }
        }
        private BehaviorProperty<string> _stopButtonText;

        /// <summary>
        /// PlayInterval
        /// </summary>
        public BehaviorProperty<int> PlayInterval
        {
            get { return _playInterval; }
        }
        private BehaviorProperty<int> _playInterval;

        /// <summary>
        /// ImageTitleLabelID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> ImageTitleLabel
        {
            get { return _imageTitleLabel; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _imageTitleLabel;

        /// <summary>
        /// ImageDescriptionLabelID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> ImageDescriptionLabel
        {
            get { return _imageDescriptionLabel; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _imageDescriptionLabel;

        /// <summary>
        /// PreviousButton
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> PreviousButton
        {
            get { return _previousButton; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _previousButton;

        /// <summary>
        /// Loop
        /// </summary>
        public BehaviorProperty<bool> Loop
        {
            get { return _loop; }
        }
        private BehaviorProperty<bool> _loop;

        /// <summary>
        /// AutoPlay
        /// </summary>
        public BehaviorProperty<bool> AutoPlay
        {
            get { return _autoPlay; }
        }
        private BehaviorProperty<bool> _autoPlay;

        /// <summary>
        /// SlideChanged
        /// </summary>
        public BehaviorEvent SlideChanged
        {
            get { return _slideChanged; }
        }
        private BehaviorEvent _slideChanged;

        /// <summary>
        /// AjaxControlToolkit.SlideShowExtender Wrapper
        /// </summary>
        /// <param name="element">Target image element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public SlideShowBehavior(HtmlImageElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _slideShowServiceMethod = BehaviorProperty<string>.CreateProperty(this, "slideShowServiceMethod");
            _slideShowServicePath = BehaviorProperty<string>.CreateProperty(this, "slideShowServicePath");
            _contextKey = BehaviorProperty<string>.CreateProperty(this, "contextKey");
            _useContextKey = BehaviorProperty<bool>.CreateProperty(this, "useContextKey");
            _nextButton = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "nextButtonID");
            _playButton = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "playButtonID");
            _playButtonText = BehaviorProperty<string>.CreateProperty(this, "playButtonText");
            _stopButtonText = BehaviorProperty<string>.CreateProperty(this, "stopButtonText");
            _playInterval = BehaviorProperty<int>.CreateProperty(this, "playInterval");
            _imageTitleLabel = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "imageTitleLabelID");
            _imageDescriptionLabel = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "imageDescriptionLabelID");
            _previousButton = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "previousButtonID");
            _loop = BehaviorProperty<bool>.CreateProperty(this, "loop");
            _autoPlay = BehaviorProperty<bool>.CreateProperty(this, "autoPlay");
            _slideChanged = BehaviorEvent<object>.CreateEvent(this, "slideChanged");
        }

        /// <summary>
        /// Reloads the image elements and returns the current image's path.
        /// </summary>
        public string CurrentImagePath
        {
            get
            {
                // Wait for the image to load for 2 seconds. If the image is loaded then the url 
                // will end with a *.jpeg or *.jpg or any other file extension name and hence the 
                // condition that the url should not end in a slash to ensure that the image was loaded.
                // Even if the image was not loaded we just return so the exception is ignored. If the user
                // is expecting no image because of bad links they can determine what action to take based on
                // the test.
                try
                {
                    Common.Wait(this.Page, 3, delegate()
                                {
                                    return !(this.Element.GetAttributes().Source.EndsWith("/"));
                                });
                }
                catch (WebTestingException)
                { }
                return this.Element.GetAttributes().Source;
            }
        }

        /// <summary>
        /// Returns the current image's alternate text
        /// </summary>
        public string CurrentImageAlternateText
        {
            get
            {
                return this.Element.GetAttributes().AlternateText;
            }
        }

        /// <summary>
        /// Returns the current image's title
        /// </summary>
        public string CurrentImageTitle
        {
            get
            {
                if (this.ImageTitleLabel.Reference != null)
                {
                    return this.ImageTitleLabel.Reference.GetAttributes().InnerText;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        /// <summary>
        /// Returns the current image's description
        /// </summary>
        public string CurrentImageDescription
        {
            get
            {
                if (this.ImageDescriptionLabel.Reference != null)
                {
                    return this.ImageDescriptionLabel.Reference.GetAttributes().InnerText;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        /// <summary>
        /// Click on the next button if available otherwise throw
        /// </summary>
        public void ClickNext()
        {
            Assert.IsNotNull(this.NextButton.Reference);
            this.NextButton.Reference.Click();
        }

        /// <summary>
        /// Click on the previous button if available otherwise throw
        /// </summary>
        public void ClickPrevious()
        {
            Assert.IsNotNull(this.PreviousButton.Reference);
            this.PreviousButton.Reference.Click();
        }

        /// <summary>
        /// Click on the play button if available otherwise throw
        /// </summary>
        public void ClickPlay()
        {
            Assert.IsNotNull(this.PlayButton.Reference);
            this.PlayButton.Reference.Click();
        }
        /// <summary>
        /// Ensure that the slide description, title and url are correctly 
        /// on the various elements attached to the slideshow extender
        /// </summary>
        public void AssertSlide(string imageUrl, string title, string description)
        {
            this.AssertSlideUrl(imageUrl);
            this.AssertSlideDescription(description);
            this.AssertSlideTitle(title);
        }

        /// <summary>
        /// Ensure that the slide is correctly set on image attached to the slideshow extender
        /// </summary>
        public void AssertSlideUrl(string imageUrl)
        {
            Assert.StringContains(this.CurrentImagePath.ToLowerInvariant(), imageUrl.ToLowerInvariant());
        }

        /// <summary>
        /// Ensure that the slide title is correctly set on title label attached to the slideshow extender
        /// </summary>
        public void AssertSlideTitle(string title)
        {
            Assert.StringEndsWith(this.CurrentImageTitle, title);
            Assert.StringEndsWith(this.CurrentImageAlternateText, title);
        }

        /// <summary>
        /// Ensure that the slide description is correctly set on description label attached to the slideshow extender
        /// </summary>
        public void AssertSlideDescription(string description)
        {
            Assert.StringEndsWith(this.CurrentImageDescription, description);
        }
    }
}