// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Web.UI;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The ReferenceBehaviorProperty models properties whose Value is actually
    /// a reference to something else.
    /// </summary>
    /// <typeparam name="R">Type of the object referenced by the Value property</typeparam>
    /// <typeparam name="T">Type of the Value property</typeparam>
    public abstract class ReferenceBehaviorProperty<R, T> : BehaviorProperty<T>
    {
        /// <summary>
        /// Object referenced by Value
        /// </summary>
        public R Reference
        {
            get
            {
                if (!Synchronized || IsVolatile || _reference == null)
                {
                    GetReference();
                }
                return _reference;
            }
        }
        protected R _reference;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="memberType">Type of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        protected ReferenceBehaviorProperty(Behavior behavior, string name, ClientMemberType memberType, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
            : base(behavior, name, memberType, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter)
        {
        }

        /// <summary>
        /// Destroy the reference whenever the Value is retrieved
        /// </summary>
        protected override void GetValue()
        {
            base.GetValue();
            _reference = default(R);
        }

        /// <summary>
        /// Destroy the reference whenever the Value is updated
        /// </summary>
        protected override void SetValue(object value)
        {
            base.SetValue(value);
            _reference = default(R);
        }

        /// <summary>
        /// Get the object corresponding to the reference
        /// </summary>
        protected abstract void GetReference();

        /// <summary>
        /// Implicitly use the property's Reference
        /// </summary>
        /// <param name="property">ReferenceBehaviorProperty</param>
        /// <returns>Reference</returns>
        public static implicit operator R(ReferenceBehaviorProperty<R, T> property)
        {
            return property.Reference;
        }
    }

    /// <summary>
    /// The ReferenceBehaviorProperty models properties that are IDs for
    /// other page elements.
    /// </summary>
    /// <typeparam name="R">Type of the element described by the property</typeparam>
    public class ReferenceBehaviorProperty<R> : ReferenceBehaviorProperty<R, string>
        where R : HtmlElement
    {
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="memberType">Type of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        protected ReferenceBehaviorProperty(Behavior behavior, string name, ClientMemberType memberType, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
            : base(behavior, name, memberType, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter)
        {
        }

        /// <summary>
        /// Get the element corresponding to the reference
        /// </summary>
        protected override void GetReference()
        {
            string id = Value;
            _reference = (!string.IsNullOrEmpty(id)) ?
                Behavior.Page.Elements.Find(id) as R :
                null;
        }

        #region Static Create Methods
        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, Converter<object, string> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, null, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, Converter<object, string> propertyConverter)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, string lookupExpression)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, string lookupExpression, Converter<object, string> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, Converter<object, string> propertyConverter)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return new ReferenceBehaviorProperty<R>(behavior, name, ClientMemberType.Property, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, Converter<object, string> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, string lookupExpression)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, string lookupExpression, Converter<object, string> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, null, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, Converter<object, string> propertyConverter)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, Converter<object, string> propertyConverter)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return new ReferenceBehaviorProperty<R>(behavior, name, ClientMemberType.Field, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateCustomProperty(Behavior behavior, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateCustomProperty(behavior, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateCustomProperty(Behavior behavior, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return CreateCustomProperty(behavior, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateCustomProperty(Behavior behavior, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateCustomProperty(behavior, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new ReferenceBehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static new ReferenceBehaviorProperty<R> CreateCustomProperty(Behavior behavior, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, string> propertyConverter)
        {
            return new ReferenceBehaviorProperty<R>(behavior, null, ClientMemberType.Custom, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }
        #endregion Static Create Methods
    }
}