// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit.Testing.Client;
using AjaxControlToolkit.Testing.Client.MaskedEdit;

/// <summary>
/// Ensure that Masked edit works fine when the mask type is time.
/// </summary>
/// <TestSuite>
///   <Path>MaskedEdit\Time.aspx</Path>
///   <ComponentType>AjaxControlToolkit.Testing.Client.MaskedEditBehavior</ComponentType>
///   <WrapperType>AjaxControlToolkit.Testing.Client.MaskedEdit.TimePage</WrapperType>
/// </TestSuite>
[WebTestClass]
[WebTestTag("MaskedEdit.Time")]
public class MaskedEdit_Time : AjaxControlToolkit.Testing.Client.MaskedEdit.MaskedEdit_Common
{
    public override IEnumerable<MaskedEditScenario> MaskedEdits
    {
        get 
        { 
            TimePage page = new TimePage();
            yield return new MaskedEditScenario(page.TimeTextBoxMaskedEditExtender, "__:__ AM", page.TimeValidator);
            yield return new MaskedEditScenario(page.CultureSensitiveMaskedEditExtender, "__:__:__", page.CultureSensitiveMaskedEditValidator);
        }
    }

    /// <summary>
    /// Make sure textbox accepts valid time and the validator is happy.
    /// </summary>
    /// <remarks>
    /// PASS
    /// </remarks>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.Safari, "Time gets cleared in blur in Safari")] 
    public void ValidTime()
    {
        TimePage page = new TimePage();
        page.TimeTextBox.SetText("12:23 PM", true);
        Assert.AreEqual("12:23 PM", page.TimeTextBox.GetAttributes().Value ?? string.Empty);
        Assert.AreEqual(string.Empty, page.TimeValidator.GetAttributes().InnerText);

    }

    /// <summary>
    /// Test to validate that Invalid time is rejected by the validator.
    /// </summary>
    /// <remarks>
    /// PASS
    /// </remarks>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7 | Browsers.Safari , "Validator message does not show up.")] 
    public void InvalidTime()
    {
        TimePage page = new TimePage();
        page.TimeTextBox.SetText("22:22 PM", true);
        Assert.AreEqual("22:22 PM", page.TimeTextBox.GetAttributes().Value ?? string.Empty);
        Assert.AreEqual("Invalid time", page.TimeValidator.GetAttributes().InnerText);

        page.CultureSensitiveMaskedEditTextBox.SetText("45:45:45", true);
        Assert.AreEqual("45:45:45", page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value ?? string.Empty);
        Assert.AreEqual("Invalid time", page.CultureSensitiveMaskedEditValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure textbox accepts valid time.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Value does not get set at all.")]
    public void InvalidCharactersTime()
    {
        TimePage page = new TimePage();
        page.TimeTextBox.SetText("aaaa22:a22b PM", true);
        Assert.AreEqual("22:22 PM", page.TimeTextBox.GetAttributes().Value ?? string.Empty);
        Assert.AreEqual("Invalid time", page.TimeValidator.GetAttributes().InnerText);

        page.CultureSensitiveMaskedEditTextBox.SetText("aa:bb:xx", true);
        Assert.AreEqual(string.Empty, page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value ?? string.Empty);
        Assert.AreEqual("Invalid time", page.TimeValidator.GetAttributes().InnerText);

    }

    /// <summary>
    /// Make sure that if no time is entered the validator complains and the mask is empty
    /// blur as expected.
    /// </summary>
    [WebTestMethod]
    public void NoTime()
    {
        TimePage page = new TimePage();
        page.TimeTextBox.Focus();
        page.TimeTextBox.Blur();
        Assert.IsNull(page.TimeTextBox.GetAttributes().Value);
        Assert.AreEqual("Please enter time", page.TimeValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Send the "P" to the time text box and make sure the mask switches to PM.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Nothing happens and the mask does not switch. It seems it is not accepting that input at all. It works fine manually.")]
    public void SwitchAMPM()
    {
        TimePage page = new TimePage();
        page.TimeTextBox.Focus();
        page.TimeTextBox.SetText("P", false);
        Assert.AreEqual("__:__ PM", page.TimeTextBox.GetAttributes().Value);
        page.TimeTextBox.Blur();
        Assert.AreEqual("Please enter time", page.TimeValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure that time greater than or equal to minimum value is allowed/
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7, "TODO: ???")]
    public void MinimumTimeLimit()
    {
        TimePage page = new TimePage();
        page.CultureSensitiveMaskedEditTextBox.SetText("03:29:59", true);
        Assert.AreEqual("03:29:59", page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value);
        Assert.AreEqual("Message Min", page.CultureSensitiveMaskedEditValidator.GetAttributes().InnerText);

        page.CultureSensitiveMaskedEditTextBox.SetText("03:30:00", true);
        Assert.AreEqual("03:30:00", page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value);
        Assert.AreEqual(string.Empty, page.CultureSensitiveMaskedEditValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure that time less than or equal to maximum value is allowed.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7, "TODO: ???")]
    public void MaximumTimeLimit()
    {
        TimePage page = new TimePage();
        page.CultureSensitiveMaskedEditTextBox.SetText("18:00:01", true);
        Assert.AreEqual("18:00:01", page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value);
        Assert.AreEqual("Message Max", page.CultureSensitiveMaskedEditValidator.GetAttributes().InnerText);

        page.CultureSensitiveMaskedEditTextBox.SetText("18:00:00", true);
        Assert.AreEqual("18:00:00", page.CultureSensitiveMaskedEditTextBox.GetAttributes().Value);
        Assert.AreEqual(string.Empty, page.CultureSensitiveMaskedEditValidator.GetAttributes().InnerText);
    }
}