// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.ListSearch
{
    /// <summary>
    /// Common ListSearch tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.ListSearchBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("ListSearch")]
    [WebTestTag("Popup")]
    public abstract class ListSearch_Common
    {
        public abstract IEnumerable<ListSearchBehavior> ListSearchBehaviors
        {
            get;
        }

        [WebTestMethod]
        public void DefaultPrompt()
        {
            foreach (ListSearchBehavior listSearchBehavior in ListSearchBehaviors)
            {
                HtmlSelectElement listElement = listSearchBehavior.Element;

                Assert.IsFalse(listSearchBehavior.PromptDivExists.Value);
                listElement.Focus();
                listSearchBehavior.Invalidate();
                listSearchBehavior.PromptDivExists.Require();
                listSearchBehavior.PromptText.Require();
                listSearchBehavior.PromptDiv.Require();
                Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                Assert.AreEqual(listSearchBehavior.PromptText.Value, listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                listElement.Blur();
                listSearchBehavior.Invalidate();
                Assert.IsFalse(listSearchBehavior.PromptDivExists.Value);
            }
        }

        [WebTestMethod]
        public void CustomPrompt()
        {
            string alternatePromptText = "Alternate text";
            foreach (ListSearchBehavior listSearchBehavior in ListSearchBehaviors)
            {
                HtmlSelectElement listElement = listSearchBehavior.Element;

                listSearchBehavior.PromptText.Value = alternatePromptText;
                listSearchBehavior.PromptPosition.Value = AjaxControlToolkit.ListSearchPromptPosition.Bottom;
                listElement.Focus();
                listSearchBehavior.Invalidate();
                Assert.AreEqual(alternatePromptText, listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                listElement.Blur();
            }
        }

        [WebTestMethod]
        [WebTestFailureTag(Browsers.Opera, "Requires support for document.createEvent(\"KeyboardEvent\").")]
        [WebTestFailureTag(Browsers.Safari, "Requires support for DispatchEvent on a KeyboardEvent.")]
        public void IncrementalTypingStartsWithQueryPattern()
        {
            foreach (ListSearchBehavior listSearchBehavior in ListSearchBehaviors)
            {
                HtmlSelectElement listElement = listSearchBehavior.Element;

                Common.TypeText(listElement, "b", true, false);
                listSearchBehavior.Invalidate();
                listSearchBehavior.PromptDivExists.Require();
                listSearchBehavior.PromptDiv.Require();
                Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                Assert.AreEqual("b", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                HtmlSelectElementAttributeReader attributes = listElement.GetAttributes();
                Assert.AreEqual(2, attributes.SelectedIndex);
                Assert.AreEqual("Bissotwo", attributes.Value);
                Common.TypeText(listElement, "\b", false, false);
                listSearchBehavior.PromptDivExists.Invalidate();
                Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                Assert.AreEqual("", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                Assert.IsNull(listElement.GetAttributes().SelectedIndex); // NEXUS bug: Actually 0
                Assert.AreEqual("Alfa", listElement.GetAttributes().Value);
                Common.TypeText(listElement, "nAd", false, false);
                Assert.AreEqual("nAd", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                attributes = listElement.GetAttributes();
                Assert.AreEqual(20, attributes.SelectedIndex);
                Assert.AreEqual("Nadazero", attributes.Value);
                Common.TypeText(listElement, "\raz", false, false);
                Assert.AreEqual("nAdaz", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                attributes = listElement.GetAttributes();
                Assert.AreEqual(20, attributes.SelectedIndex);
                Assert.AreEqual("Nadazero", attributes.Value);
                listElement.Click();
                Common.TypeText(listElement, "\b\b\b\bO", false, false);
                Assert.AreEqual("nO", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                attributes = listElement.GetAttributes();
                Assert.AreEqual(22, attributes.SelectedIndex);
                Assert.AreEqual("November", attributes.Value);
                Common.TypeText(listElement, "&", false, false);
                Assert.AreEqual("nO&", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                attributes = listElement.GetAttributes();
                Assert.AreEqual(22, attributes.SelectedIndex);
                Assert.AreEqual("November", attributes.Value);
                Common.TypeText(listElement, "\x1b", false, false);
                Assert.AreEqual("", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                Assert.IsNull(listElement.GetAttributes().SelectedIndex); // NEXUS bug: Actually 0
                Assert.AreEqual("Alfa", listElement.GetAttributes().Value);
                listElement.Blur();
            }
        }

        [WebTestMethod]
        [WebTestFailureTag(Browsers.Opera, "Requires support for document.createEvent(\"KeyboardEvent\").")]
        [WebTestFailureTag(Browsers.Safari, "Requires support for DispatchEvent on a KeyboardEvent.")]
        public void IncrementalTypingContainsQueryPattern()
        {
            foreach (ListSearchBehavior listSearchBehavior in ListSearchBehaviors)
            {
                if (listSearchBehavior.QueryPattern.Value == ListSearchQueryPattern.Contains)
                {
                    HtmlSelectElement listElement = listSearchBehavior.Element;

                    Common.TypeText(listElement, "iss", true, false);
                    listSearchBehavior.Invalidate();
                    listSearchBehavior.PromptDivExists.Require();
                    listSearchBehavior.PromptDiv.Require();
                    Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                    Assert.AreEqual("iss", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    HtmlSelectElementAttributeReader attributes = listElement.GetAttributes();
                    Assert.AreEqual(2, attributes.SelectedIndex);
                    Assert.AreEqual("Bissotwo", attributes.Value);
                    Common.TypeText(listElement, "\b\b\b", false, false);
                    listSearchBehavior.PromptDivExists.Invalidate();
                    Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                    Assert.AreEqual("", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    Assert.IsNull(listElement.GetAttributes().SelectedIndex); // NEXUS bug: Actually 0
                    Assert.AreEqual("Alfa", listElement.GetAttributes().Value);
                    Common.TypeText(listElement, "zer", false, false);
                    Assert.AreEqual("zer", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    attributes = listElement.GetAttributes();
                    Assert.AreEqual(20, attributes.SelectedIndex);
                    Assert.AreEqual("Nadazero", attributes.Value);
                    listElement.Click();
                    Common.TypeText(listElement, "\b\b\beMber", false, false);
                    Assert.AreEqual("eMber", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    attributes = listElement.GetAttributes();
                    Assert.AreEqual(22, attributes.SelectedIndex);
                    Assert.AreEqual("November", attributes.Value);
                    Common.TypeText(listElement, "&", false, false);
                    Assert.AreEqual("eMber&", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    attributes = listElement.GetAttributes();
                    Assert.AreEqual(22, attributes.SelectedIndex);
                    Assert.AreEqual("November", attributes.Value);
                    Common.TypeText(listElement, "\x1b", false, false);
                    Assert.AreEqual("", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                    Assert.IsNull(listElement.GetAttributes().SelectedIndex); // NEXUS bug: Actually 0
                    Assert.AreEqual("Alfa", listElement.GetAttributes().Value);
                    listElement.Blur();
                }
            }
        }

        [WebTestMethod]
        [WebTestFailureTag(Browsers.Opera, "Requires support for document.createEvent(\"KeyboardEvent\").")]
        [WebTestFailureTag(Browsers.Safari, "Requires support for DispatchEvent on a KeyboardEvent.")]
        public void QueryTimeout()
        {
            foreach (ListSearchBehavior listSearchBehavior in ListSearchBehaviors)
            {
                HtmlSelectElement listElement = listSearchBehavior.Element;

                if (listSearchBehavior.QueryTimeout.Value > 0)
                {
                    // Search for something that does not exist and make sure that the
                    // list prompt goes away after the QueryTimeout
                    Common.TypeText(listElement, "2141324", true, false);
                    listSearchBehavior.Invalidate();
                    listSearchBehavior.PromptDivExists.Require();
                    listSearchBehavior.PromptDiv.Require();
                    Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                    Common.Wait(listSearchBehavior.Page, (listSearchBehavior.QueryTimeout.Value / 1000),
                        delegate()
                        {
                            Assert.StringIsNullOrEmpty(listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                            return true;
                        });


                    // search for something that results in a match and make sure that 
                    // the query prompt does not go away after the query prompt
                    Common.TypeText(listElement, "nAd", true, false);
                    listSearchBehavior.Invalidate();
                    listSearchBehavior.PromptDivExists.Require();
                    listSearchBehavior.PromptDiv.Require();
                    Assert.IsTrue(listSearchBehavior.PromptDivExists.Value);
                    Common.Wait(listSearchBehavior.Page, (listSearchBehavior.QueryTimeout.Value / 1000),
                        delegate()
                        {
                            Assert.AreEqual("nAd", listSearchBehavior.PromptDiv.Reference.GetAttributes().InnerText);
                            return true;
                        });
                    HtmlSelectElementAttributeReader attributes = listElement.GetAttributes();
                    Assert.AreEqual(20, attributes.SelectedIndex);
                    Assert.AreEqual("Nadazero", attributes.Value);
                }
            }
        }
    }
}
