// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.CascadingDropDown Model
    /// </summary>
    /// <TestComponent Name="CascadingDropDown">
    ///   <ToolkitType>AjaxControlToolkit.CascadingDropDown</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.CascadingDropDown.CascadingDropDown_Common</CommonTestSuite>
    /// </TestComponent>
    public class CascadingDropDownBehavior : Behavior<HtmlSelectElement>
    {
        /// <summary>
        /// ParentControlID
        /// </summary>
        public BehaviorProperty<string> ParentControlID
        {
            get { return _parentControlID; }
        }
        private BehaviorProperty<string> _parentControlID;

        /// <summary>
        /// Category
        /// </summary>
        public BehaviorProperty<string> Category
        {
            get { return _category; }
        }
        private BehaviorProperty<string> _category;

        /// <summary>
        /// PromptText
        /// </summary>
        public BehaviorProperty<string> PromptText
        {
            get { return _promptText; }
        }
        private BehaviorProperty<string> _promptText;

        /// <summary>
        /// PromptValue
        /// </summary>
        public BehaviorProperty<string> PromptValue
        {
            get { return _promptValue; }
        }
        private BehaviorProperty<string> _promptValue;

        /// <summary>
        /// EmptyText
        /// </summary>
        public BehaviorProperty<string> EmptyText
        {
            get { return _emptyText; }
        }
        private BehaviorProperty<string> _emptyText;

        /// <summary>
        /// EmptyValue
        /// </summary>
        public BehaviorProperty<string> EmptyValue
        {
            get { return _emptyValue; }
        }
        private BehaviorProperty<string> _emptyValue;

        /// <summary>
        /// LoadingText
        /// </summary>
        public BehaviorProperty<string> LoadingText
        {
            get { return _loadingText; }
        }
        private BehaviorProperty<string> _loadingText;

        /// <summary>
        /// SelectedValue
        /// </summary>
        public BehaviorProperty<string> SelectedValue
        {
            get { return _selectedValue; }
        }
        private BehaviorProperty<string> _selectedValue;

        /// <summary>
        /// ServicePath
        /// </summary>
        public BehaviorProperty<string> ServicePath
        {
            get { return _servicePath; }
        }
        private BehaviorProperty<string> _servicePath;

        /// <summary>
        /// ServiceMethod
        /// </summary>
        public BehaviorProperty<string> ServiceMethod
        {
            get { return _serviceMethod; }
        }
        private BehaviorProperty<string> _serviceMethod;

        /// <summary>
        /// ContextKey
        /// </summary>
        public BehaviorProperty<string> ContextKey
        {
            get { return _contextKey; }
        }
        private BehaviorProperty<string> _contextKey;

        /// <summary>
        /// UseContextKey
        /// </summary>
        public BehaviorProperty<bool> UseContextKey
        {
            get { return _useContextKey; }
        }
        private BehaviorProperty<bool> _useContextKey;

        public BehaviorProperty<int> SelectedIndex
        {
            get { return _selectedIndex; }
        }
        private BehaviorProperty<int> _selectedIndex;

        public BehaviorProperty<bool> Disabled
        {
            get { return _disabled; }
        }
        private BehaviorProperty<bool> _disabled;

        /// <summary>
        /// SelectionChanged
        /// </summary>
        public BehaviorEvent<Tuple<string, string>> SelectionChanged
        {
            get { return _selectionChanged; }
        }
        private BehaviorEvent<Tuple<string, string>> _selectionChanged;

        /// <summary>
        /// AjaxControlToolkit.CascadingDropDown Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public CascadingDropDownBehavior(HtmlSelectElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _parentControlID = BehaviorProperty<string>.CreateProperty(this, "ParentControlID");
            _category = BehaviorProperty<string>.CreateProperty(this, "Category");
            _promptText = BehaviorProperty<string>.CreateProperty(this, "PromptText");
            _promptValue = BehaviorProperty<string>.CreateProperty(this, "PromptValue");
            _emptyText = BehaviorProperty<string>.CreateProperty(this, "EmptyText");
            _emptyValue = BehaviorProperty<string>.CreateProperty(this, "EmptyValue");
            _loadingText = BehaviorProperty<string>.CreateProperty(this, "LoadingText");
            _selectedValue = BehaviorProperty<string>.CreateProperty(this, "SelectedValue");
            _servicePath = BehaviorProperty<string>.CreateProperty(this, "ServicePath");
            _serviceMethod = BehaviorProperty<string>.CreateProperty(this, "ServiceMethod");
            _contextKey = BehaviorProperty<string>.CreateProperty(this, "contextKey");
            _useContextKey = BehaviorProperty<bool>.CreateProperty(this, "useContextKey");

            _selectedIndex = BehaviorProperty<int>.CreateField(this, "selectedIndex", "{0}._element", delegate(object value) { return Common.ConvertJsonValue<int>(value ?? 0); });
            _disabled = BehaviorProperty<bool>.CreateField(this, "disabled", "{0}._element");
            _selectionChanged = BehaviorEvent<Tuple<string, string>>.CreateEvent(this,
                "selectionChanged",
                "function(sender, eventArgs) { return { oldValue:eventArgs.get_oldValue(), newValue:eventArgs.get_newValue() }; }",
                ConvertSelectionChangedEventArgs);
        }

        /// <summary>
        /// Set selectedIndex and invalidate child properties
        /// </summary>
        /// <param name="index">selected index</param>
        /// <param name="dependencies">children</param>
        public void SetSelectedIndex(int index, params CascadingDropDownBehavior[] dependencies)
        {
            Element.SetSelectedIndex(index);
            Invalidate(SelectedIndex);
            foreach(CascadingDropDownBehavior dependency in dependencies)
            {
                dependency.Invalidate(dependency.Disabled, dependency.SelectedIndex);
            }
        }

        /// <summary>
        /// Convert SelectionChangedEventArgs into a Tuple of strings
        /// </summary>
        /// <param name="value">Object deserialized from json</param>
        /// <returns>Event args</returns>
        private static Tuple<string, string> ConvertSelectionChangedEventArgs(object value)
        {
            return (null == value) ? null : new Tuple<string, string>(Common.GetJson<string>(value, "oldValue"), Common.GetJson<string>(value, "newValue"));
        }

        /// <summary>
        /// Grab all option element .text and .values for the DropDownList
        /// </summary>
        /// <returns>array of OptionInformation</returns>
        public OptionInformation[] GetOptionInformations()
        {
            IDictionary<string, object> clientOptions = Page.ExecuteScript(
                "(function() {" +
                "    var e = $get('" + Element.Id + "');" +
                "    var r = {};" +
                "    for (i = 0 ; i < e.options.length ; i++) {" +
                "        r['o' + i] = { text:e.options[i].text, value:e.options[i].value };" +
                "    }" +
                "    return r;" +
                "})()") as IDictionary<string, object>;
            Assert.IsNotNull(clientOptions);
            List<OptionInformation> OptionInformations = new List<OptionInformation>();
            for (int i = 0; i < clientOptions.Keys.Count; i++)
            {
                object clientOption = Common.GetJson<object>(clientOptions, "o" + i);
                OptionInformations.Add(new OptionInformation(Common.GetJson<string>(clientOption, "text"), Common.GetJson<string>(clientOption, "value")));
            }
            return OptionInformations.ToArray();
        }

        /// <summary>
        /// Asserts that the options of a DropDownList are correct
        /// </summary>
        /// <param name="selectedIndex">expected selectedIndex</param>
        /// <param name="expectedItems">expected items</param>
        public void AssertPopulationCorrect(int selectedIndex, params OptionInformation[] expectedItems)
        {
            Require(SelectedIndex, PromptText, PromptValue);
            Assert.AreEqual(selectedIndex, SelectedIndex.Value);
            List<OptionInformation> expected = new List<OptionInformation>(expectedItems);
            if (!string.IsNullOrEmpty(PromptText.Value))
            {
                expected.Insert(0, new OptionInformation(PromptText.Value, PromptValue.Value));
            }
            OptionInformation[] actual = GetOptionInformations();
            Assert.AreEqual(expected.Count, actual.Length);
            for (int i = 0; i < expected.Count; i++)
            {
                Assert.AreEqual(expected[i].Text, actual[i].Text);
                Assert.AreEqual(expected[i].Value ?? "", actual[i].Value ?? "");
            }
        }

        /// <summary>
        /// Assert that the DropDownList is disabled
        /// </summary>
        public void AssertDisabled()
        {
            Require(Disabled);
            Assert.IsTrue(Disabled.Value);
            AssertPopulationCorrect(0);
        }
    }
}
