// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Drawing;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.DragPanelExtender Model
    /// </summary>
    /// <TestComponent Name="DragPanel">
    ///   <ToolkitType>AjaxControlToolkit.DragPanelExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.DragPanel.DragPanel_Common</CommonTestSuite>
    /// </TestComponent>
    public class DragPanelBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// DragHandle
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DragHandle
        {
            get { return _dragHandle; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _dragHandle;

        /// <summary>
        /// ProfileProperty
        /// </summary>
        public BehaviorProperty<string> ProfileProperty
        {
            get { return _profileProperty; }
        }
        private BehaviorProperty<string> _profileProperty;

        /// <summary>
        /// Location
        /// </summary>
        public BehaviorProperty<Point> Location
        {
            get { return _location; }
        }
        private BehaviorProperty<Point> _location;

        /// <summary>
        /// DragDataType
        /// </summary>
        public BehaviorProperty<string> DragDataType
        {
            get { return _dragDataType; }
        }
        private BehaviorProperty<string> _dragDataType;

        /// <summary>
        /// DragMode
        /// </summary>
        public BehaviorProperty<int> DragMode
        {
            get { return _dragMode; }
        }
        private BehaviorProperty<int> _dragMode;

        /// <summary>
        /// DropTarget
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DropTarget
        {
            get { return _dropTarget; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _dropTarget;

        /// <summary>
        /// AjaxControlToolkit.DragPanelExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public DragPanelBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _dragHandle = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}.get_handle()");
            _profileProperty = BehaviorProperty<string>.CreateProperty(this, "profileProperty");
            _location = BehaviorProperty<Point>.CreateProperty(this, "location", null, "(function() {{ var l = {0}.get_location(); return {{ 'x': l.x, 'y': l.y }}; }})()", null,
                delegate(object value) { return new Point(Common.GetJson<int>(value, "x"), Common.GetJson<int>(value, "y")); });
            _dragDataType = BehaviorProperty<string>.CreateProperty(this, "dragDataType");
            _dragMode = BehaviorProperty<int>.CreateProperty(this, "dragMode");
            _dropTarget = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}.get_dropTargetElement()");
        }

        /// <summary>
        /// Drag the panel to a specific location
        /// </summary>
        /// <param name="from">Point to start dragging from</param>
        /// <param name="to">Point to drag it to</param>
        public void Drag(Point from, Point to)
        {
            Page.ExecuteScript("(function() { " +
                    "var b = " + BehaviorReferenceExpression + "; " +
                    "window._event = { }; " +
                    "window._event.clientX = " + from.X + "; " +
                    "window._event.clientY = " + from.Y + "; " +
                    "b.startDragDrop(b.get_element()); " +
                    "$common.setLocation(b.get_element(), {x: " + to.X + ", y: " + to.Y + "}); " +
                    "b.onDragEnd(false); " +
                "})()");
        }
    }
}