// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The Behavior class (and strongly typed Behavior&lt;T&gt; class)
    /// models an instance of a client-side ASP.NET AJAX behavior.
    /// </summary>
    public abstract class Behavior
    {
        /// <summary>
        /// Reference to the target element of the behavior
        /// </summary>
        public HtmlElement Element
        {
            get { return _element; }
            protected set { _element = value; }
        }
        private HtmlElement _element;

        /// <summary>
        /// ID used to find the behavior on the page
        /// </summary>
        public string BehaviorID
        {
            get { return _behaviorID; }
        }
        string _behaviorID;

        /// <summary>
        /// JavaScript expression used to reference the behavior
        /// </summary>
        public string BehaviorReferenceExpression
        {
            get { return _behaviorReferenceExpression; }
        }
        private string _behaviorReferenceExpression;

        /// <summary>
        /// Reference to the test page model containing the behavior
        /// </summary>
        public ToolkitTestPage Page
        {
            get { return _page; }
        }
        private ToolkitTestPage _page;

        /// <summary>
        /// List of properties associated with the behavior (this will be
        /// populated automatically with the creation of
        /// BehaviorProperty&lt;T&gt;s).
        /// </summary>
        public IList<BehaviorProperty> Properties
        {
            get { return _properties; }
        }
        private List<BehaviorProperty> _properties;

        /// <summary>
        /// Create a new behavior
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">ID of the behavior</param>
        /// <param name="page">Parent page model</param>
        public Behavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : this(element, behaviorID, null, page)
        {
        }

        /// <summary>
        /// Create a new behavior
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">ID of the behavior</param>
        /// <param name="behaviorReferenceExpression">JavaScript expression used to reference the behavior</param>
        /// <param name="page">Parent page model</param>
        public Behavior(HtmlElement element, string behaviorID, string behaviorReferenceExpression, ToolkitTestPage page)
        {
            Assert.IsNotNull(element);
            _element = element;

            _behaviorID = behaviorID;
            _behaviorReferenceExpression = string.Format(
                (!string.IsNullOrEmpty(behaviorReferenceExpression) ?
                    behaviorReferenceExpression :
                    "$find(\"{0}\")"),
                behaviorID);

            Assert.IsNotNull(page);
            _page = page;
            _page.Behaviors.Add(this);

            _properties = new List<BehaviorProperty>();
        }

        /// <summary>
        /// Require all of the properties passed as params if any or
        /// all properties associated with the behavior.
        /// </summary>
        /// <param name="properties">properties to require</param>
        public void Require(params BehaviorProperty[] properties)
        {
            if (properties == null || properties.Length == 0)
            {
                properties = _properties.ToArray();
            }
            foreach (BehaviorProperty property in properties)
            {
                property.Require();
            }
        }

        /// <summary>
        /// Invalidate all of the properties passed as params if any or
        /// all properties associated with the behavior.
        /// </summary>
        /// <param name="properties">properties to invalidate</param>
        public void Invalidate(params BehaviorProperty[] properties)
        {
            if (properties == null || properties.Length == 0)
            {
                properties = _properties.ToArray();
            }
            foreach (BehaviorProperty property in properties)
            {
                property.Invalidate();
            }
        }

        /// <summary>
        /// Invalidate and Require all of the properties passed as params if any or
        /// all properties associated with the behavior.
        /// </summary>
        /// <param name="properties">properties to invalidate and require</param>
        public void InvalidateAndRequire(params BehaviorProperty[] properties)
        {
            this.Invalidate(properties);
            this.Require(properties);
        }
    }

    /// <summary>
    /// Strongly-typed Behavior
    /// </summary>
    /// <typeparam name="T">Target type of the behavior</typeparam>
    public abstract class Behavior<T> : Behavior
        where T : HtmlElement
    {
        /// <summary>
        /// Constructor to create a new behavior
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">ID of the behavior</param>
        /// <param name="page">Parent page model</param>
        public Behavior(T element, string behaviorID, ToolkitTestPage pageWrapper)
            : this(element, behaviorID, null, pageWrapper)
        {
        }

        /// <summary>
        /// Constructor to create a new behavior
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">ID of the behavior</param>
        /// <param name="behaviorReferenceExpression">JavaScript expression used to reference the behavior</param>
        /// <param name="page">Parent page model</param>
        public Behavior(T element, string behaviorID, string behaviorReferenceExpression, ToolkitTestPage pageWrapper)
            : base(element, behaviorID, behaviorReferenceExpression, pageWrapper)
        {
        }

        /// <summary>
        /// Reference to the target element of the behavior
        /// </summary>
        public new T Element
        {
            get { return base.Element as T; }
        }
    }
}