<?php

declare(strict_types=1);

namespace popp\r24\zestaw01\parse;

/* listing 24.01 */
class Scanner
{
    // rodzaje elementów leksykalnych
    public const WORD         = 1;
    public const QUOTE        = 2;
    public const APOS         = 3;
    public const WHITESPACE   = 6;
    public const EOL          = 8;
    public const CHAR         = 9;
    public const EOF          = 0;
    public const SOF          = -1;

    protected int $line_no    = 1;
    protected int $char_no    = 0;
    protected ?string $token  = null;
    protected int $token_type     = -1;

    // Reader daje dostęp do "surowych" danych, Context 
    // przechowuje dane wynikowe
    public function __construct(private Reader $r, private Context $context)
    {
    }

    public function getContext(): Context
    {
        return $this->context;
    }

    // pominięcie znaków odstępu
    public function eatWhiteSpace(): int
    {
        $ret = 0;

        if (
            $this->token_type != self::WHITESPACE &&
            $this->token_type != self::EOL
        ) {
            return $ret;
        }

        while (
            $this->nextToken() == self::WHITESPACE ||
            $this->token_type == self::EOL
        ) {
            $ret++;
        }

        return $ret;
    }

    // pobranie symbolicznej reprezentacji elementu leksykalnego
    // (bieżącego albo wskazanego argumentem $int
    public function getTypeString(int $int = -1): ?string
    {
        if ($int < 0) {
            $int = $this->tokenType();
        }

        if ($int < 0) {
            return null;
        }

        $resolve = [
            self::WORD =>       'WORD',
            self::QUOTE =>      'QUOTE',
            self::APOS =>       'APOS',
            self::WHITESPACE => 'WHITESPACE',
            self::EOL =>        'EOL',
            self::CHAR =>       'CHAR',
            self::EOF =>        'EOF'
        ];

        return $resolve[$int];
    }

    // bieżący rodzaj elementu (reprezentowany liczbowo)
    public function tokenType(): int
    {
        return $this->token_type;
    }

    // zawartość bieżącego elementu
    public function token(): ?string
    {
        return $this->token;
    }

    // zwraca true, jeśli bieżący element to słowo
    public function isWord(): bool
    {
        return ($this->token_type == self::WORD);
    }

    // zwraca true, jeśli bieżący element to znak cudzysłowu
    public function isQuote(): bool
    {
        return ($this->token_type == self::APOS || $this->token_type == self::QUOTE);
    }

    // numer bieżącego wiersza w strumieniu źródłowym
    public function lineNo(): int
    {
        return $this->line_no;
    }

    // numer bieżącego znaku w strumieniu źródłowym
    public function charNo(): int
    {
        return $this->char_no;
    }

    // klonowanie obiektu
    public function __clone(): void
    {
        $this->r = clone($this->r);
    }

    // Przejście do następnego elementu w strumieniu. Ustawienie
    // bieżącego elementu i zapamiętanie numeru wiersza i znaku
    public function nextToken(): int
    {
        $this->token = null;
        $type = -1;

        while (! is_bool($char = $this->getChar())) {
            if ($this->isEolChar($char)) {
                $this->token = $this->manageEolChars($char);
                $this->line_no++;
                $this->char_no = 0;

                return ($this->token_type = self::EOL);
            } elseif ($this->isWordChar($char)) {
                $this->token = $this->eatWordChars($char);
                $type = self::WORD;
            } elseif ($this->isSpaceChar($char)) {
                $this->token = $char;
                $type = self::WHITESPACE;
            } elseif ($char == "'") {
                $this->token = $char;
                $type = self::APOS;
            } elseif ($char == '"') {
                $this->token = $char;
                $type = self::QUOTE;
            } else {
                $type = self::CHAR;
                $this->token = $char;
            }

            $this->char_no += strlen($this->token());

            return ($this->token_type = $type);
        }

        return ($this->token_type = self::EOF);
    }

    // zwraca tablicę (typ elementu i zawartość elementu)
    // dla NASTĘPNEGO elementu w strumieniu
    public function peekToken(): array
    {
        $state = $this->getState();
        $type = $this->nextToken();
        $token = $this->token();
        $this->setState($state);

        return [$type, $token];
    }

    // pobranie obiektu ScannerState reprezentującego bieżącą 
    // pozycję skanera w strumieniu i dane o bieżącym elemencie
    public function getState(): ScannerState
    {
        $state = new ScannerState();
        $state->line_no      = $this->line_no;
        $state->char_no      = $this->char_no;
        $state->token        = $this->token;
        $state->token_type   = $this->token_type;
        $state->r            = clone($this->r);
        $state->context      = clone($this->context);

        return $state;
    }

    // użycie obiektu ScannerState do przywrócenia stanu skanera
    public function setState(ScannerState $state): void
    {
        $this->line_no      = $state->line_no;
        $this->char_no      = $state->char_no;
        $this->token        = $state->token;
        $this->token_type   = $state->token_type;
        $this->r            = $state->r;
        $this->context      = $state->context;
    }

    // pobranie następnego znaku ze strumienia źródłowego
    // zwraca wartość logiczną, jeśli nie ma więcej znaków
    private function getChar(): string|bool
    {
        return $this->r->getChar();
    }

    // pobieranie kolejnych znaków, dopóki są znakami słowa
    private function eatWordChars(string $char): string
    {
        $val = $char;

        while ($this->isWordChar($char = $this->getChar())) {
            $val .= $char;
        }

        if ($char) {
            $this->pushBackChar();
        }

        return $val;
    }

    // cofnięcie się o jeden znak w strumieniu źródłowym
    private function pushBackChar(): void
    {
        $this->r->pushBackChar();
    }

    // argument jest znakiem słowa
    private function isWordChar($char): bool
    {
        if (is_bool($char)) {
            return false;
        }

        return (preg_match("/[A-Za-z0-9_\-]/", $char) === 1);
    }

    // argument jest znakiem odstępu
    private function isSpaceChar($char): bool
    {
        return (preg_match("/\t| /", $char) === 1);
    }

    // argument jest znakiem końca wiersza
    private function isEolChar($char): bool
    {
        $check = preg_match("/\n|\r/", $char);

        return ($check === 1);
    }

    // połknięcie znaków \n, \r bądź \r\n
    private function manageEolChars(string $char): string
    {
        if ($char == "\r") {
            $next_char = $this->getChar();

            if ($next_char == "\n") {
                return "{$char}{$next_char}";
            } else {
                $this->pushBackChar();
            }
        }

        return $char;
    }

    public function getPos(): int
    {
        return $this->r->getPos();
    }
}
/* /listing 24.01 */
