<?php

class Scanner {
    const WORD         = 1;
    const QUOTE        = 2;
    const APOS         = 3;
    const WHITESPACE   = 6;
    const EOL          = 8;
    const CHAR         = 99;
    const EOF          = 0;

    protected $in;
    protected $line_no = 1;
    protected $char_no = 0;
    protected $token;
    protected $token_type;
    protected $regexps;
    public $resultstack = array();

    function __construct( $in ) {
        $this->in = $in;
        $this->setRegexps();
        $this->nextToken();
        $this->eatWhiteSpace();
    }

    // od wynik na stos resultstack
    function pushResult( $mixed ) {
        array_push( $this->resultstack, $mixed );
    }

    // usu wynik ze stosu resultstack
    function popResult( ) {
        return array_pop( $this->resultstack );
    }

    // podaj liczb llementw na stosie resultstack
    function resultCount() {
        return count( $this->resultstack );
    }

    // zwr ostatni element ze stosu resultstack, nie zdejmujc go z tego stosu 
    function peekResult( ) {
        if ( empty( $this->resultstack ) ) {
            throw new Exception( "stos resultstack pusty" );
        }
        return $this->resultstack[count( $this->resultstack ) -1 ];
    }

    // zainicjalizuj tablic wyrae regularnych reprezentujcych elementy leksykalne
    private function setRegexps() {
        $this->regexps = array( 
                      self::WHITESPACE => '[ \t]', 
                      self::EOL => '\n', 
                      self::WORD => '[a-zA-Z0-9_-]+\b',
                      self::QUOTE => '"',
                      self::APOS => "'",
        );

        $this->typestrings = array( 
                      self::WHITESPACE => 'WHITESPACE', 
                      self::EOL => 'EOL', 
                      self::WORD => 'WORD',
                      self::QUOTE => 'QUOTE',
                      self::APOS => "APOS",
                      self::CHAR  => 'CHAR',
                      self::EOF => 'EOF'
        );
    }

    // pomi znaki odstpw
    function eatWhiteSpace( ) {
        $ret = 0;
        if ( $this->token_type != self::WHITESPACE && 
             $this->token_type != self::EOL ) {
            return $ret;
        }
        while ( $this->nextToken() == self::WHITESPACE || 
                $this->token_type == self::EOL ) {
            $ret++;
        }
        return $ret;
    }

    // dla podanej staej zwr jej wersj napisow
    // np. 1 => 'WORD',
    function getTypeString( $int=-1 ) {
        if ( $int<0 ) {
            $int=$this->token_type(); 
        }
        return $this->typestrings[$int];
    }

    // podaj typ biecego elementu
    function token_type() {
        return $this->token_type;
    }

    // skanowany tekst skraca si w miar wyodrbniania z niego
    // kolejnych elementw leksykalnych
    function input() {
        return $this->in;
    }

    // zwr biecy element
    function token() {
        return $this->token;
    }

    // zwr numer biecego wiersza (przydatne w komunikatach o bdach)
    function line_no() {
        return $this->line_no;
    }

    // zwr numer biecego znaku
    function char_no() {
        return $this->char_no;
    }

    // sprbuj wyodrbni z wejcia nastpny element leksykalny;
    // jeli nie uda si dopasowa obszerniejszego elementu, otrzymamy element znakowy
    function nextToken() {
        if ( ! strlen( $this->in ) ) {
            return ( $this->token_type = self::EOF );
        }

        $ret = 0;
        foreach ( $this->regexps as $type=>$regex ) {        
            if ( $ret = $this->testToken( $regex, $type ) ) {
                if ( $ret == self::EOL ) {
                    $this->line_no++;
                    $this->char_no = 0;
                } else {
                    $this->char_no += strlen( $this->token() );
                }
                return $ret;
            }
        }
        $this->token = substr( $this->in, 0, 1 );
        $this->in    = substr( $this->in, 1 );
        $this->char_no += 1;
        return ( $this->token_type = self::CHAR );
    }

    // dopasowanie wyraenia regularnego
    private function testToken( $regex, $type ) {
        $matches = array();
        if ( preg_match( "/^($regex)(.*)/s", $this->in, $matches ) ) {
            $this->token = $matches[1];
            $this->in    = $matches[2];
            return ( $this->token_type  = $type );
        }
        return 0;
    }

    // sklonowanie obiektu skanera
    function updateToMatch( Scanner $other ) {
        $this->in = $other->in;
        $this->token = $other->token;
        $this->token_type = $other->token_type;
        $this->char_no = $other->char_no;
        $this->line_no = $other->line_no;
        $this->resultstack = $other->resultstack;
    }
}
?>
