﻿<?php

class DebugException extends Exception
{
    const PLAIN = 0;
    const HTML  = 1;

    public static $sourceCodeSpan  = 10;
    public static $outputFormat    = self::HTML;
    public static $errorScope      = E_ERROR;

    protected $addedDebug    = array();

    public function __construct($message, $code = 0)
    {
        // sprawdza prawidowo wszelkich przypisa
        parent::__construct($message, $code);

        // dodatkowe informacje diagnostyczne?
        if (func_num_args() > 2) {
            $this->addedDebug = array_slice(func_get_args(), 2);
        }
    }

    public function __destruct()
    {
        // tre kostruktora rozmylnie pusta
    }

    // wywoywana statycznie tylko raz, w celu obsugi wszystkich bdw i wyjtkw
    public static function init()
    {
        // przypisuje klas obsugujc wyjtki
        set_exception_handler(array('DebugException', 'exceptionHandler'));
        
        // przypisuje klas obsugujc bdy
        set_error_handler(array('DebugException', 'errorHandler'), self::$errorScope);
        
        // automatyczne wykrywanie poadanego formatu generowanej treci
        if (php_sapi_name() == 'cli') {

            // jeeli wywoanie z wiersza polece to format tekstowy
            self::$outputFormat = self::PLAIN;

        } else {

            // w innym wypadku format HTML
            self::$outputFormat = self::HTML;
        }
    }

    // wyrejestrowuje klasy obsugujce wyjtki i bdy
    public static function unInit()
    {
        // zdejmuje warto ze stosu programw obsugi wyjtkw 
        restore_exception_handler();

        // zdejmuje warto ze stosu programw obsugi bdw
        restore_error_handler();
    }

    // zmienia bdy w wyjtki w formacie DebugExceptions
    public static function errorHandler($number,
                                        $message,
                                        $file,
                                        $line,
                                        $context)
    {
        // zmienia bd w wyjtek i generuje ten wyjtek
        $debugException = new DebugException($number, 0, $context);

        // przekazuje informacje do DebugException
        $debugException->file = $file;
        $debugException->line = $line;
        
        // generuje nowy wyjtek DebugException
        throw $debugException;
     }

    // przechwytuje normalne wyjtki
    public static function exceptionHandler($exception)
    {
        // jawnie wywouj metod __toString() dla biecej klasy
        self::output($exception);
    }
    
    // gromadzi i wywietla informacje diagnostyczne
    public static function output(Exception $exception)
    {
        $output = array();

        // wywietla nazw pliku i numer wiersza
        $output[] = array('Podsumowanie:', 'Wyjtek wystpi w pliku ' . basename($exception->getFile())
                    . ' w wierszu ' . $exception->getLine() . '.');
                    
        // generuje komunikat
        $output[] = array('Komunikat o bdzie: ', $exception->getMessage());
        
        // pobiera kod rdowy pliku, ktry wygenerowa wyjtek
        $sourceExcerpt = self::getSourceExcerpt($exception->getFile(), $exception->getLine());
        
        $output[] = 'Fragment kodu rdowego od wiersza ' . $sourceExcerpt['start']
                    . ' do wiersza ' . $sourceExcerpt['end'] . ' w pliku ' . $exception->getFile() . ':';

        // wyrnianie skadni dla formatu HTML
        if (self::$outputFormat == self::HTML) {
            $output[] = array('', highlight_string(implode('', $sourceExcerpt['source']), TRUE));
        } elseif (self::$outputFormat == self::PLAIN) {
            $output[] = implode('', $sourceExcerpt['source']);
        }
        
        // przejrzyste formatowanie ladu zwrotnego
        $formattedTraces = self::getFormattedTrace($exception);
        
        // docza przejrzycie sformatowane informacje diagnostyczne
        $output = array_merge($output, self::getFormattedDebugInfo($exception));

        // formatuje tre w zalenoci od wartoci $outputFormat
        // najpierw generuje HTML
        if (self::$outputFormat == self::HTML) {

            // tworzy odnonik do pokazywania i ukrywania kadego elementu ladu zwrotnego
            for ($i = 0; $i < sizeof($formattedTraces); $i++) {
                $output[] = '<a href="" onclick="var bt = document.getElementById(\'backtrace' . ($i + 1) . '\');if (bt.style.display == \'\') bt.style.display = \'none\';else bt.style.display = \'\';return false;">Backtrace step ' . ($i + 1) . ' (click to toggle):</a>';
                $output[] = self::arrayToTable($formattedTraces[$i], 'backtrace' . ($i + 1));
            }

            echo self::arrayToTable($output, null, 'Informacje diagnostyczne', FALSE);

        // generuje tre w formacie tekstowym
        } elseif (self::$outputFormat == self::PLAIN) {
        
            // docza elementy ladu zwrotnego do tablicy generowanych informacji
            $output = array_merge($output, $formattedTraces);

            // spaszcza wielowymiarow tablic na potrzeby formatu tekstowego
            $flattenedOutput = self::flattenArray($output);

            echo implode(PHP_EOL, $flattenedOutput);
        }
    }
    
    // pobiera zdefinowan przed $sourceCodeSpan ilo wierszy przed i po wierszu
    // $line z pliku $file
    public static function getSourceExcerpt($file, $line)
    {
        // pobiera plik rdowy z pliku, ktry wygenerowa wyjtek
        $source = file($file);

        // ogranicza listing do +/- liczby wierszy wskazywanej przez $sourceCodeSpan
        $startLine = max(0, $line - self::$sourceCodeSpan - 1);
        $offset = min(2 * self::$sourceCodeSpan + 1, count($source) - $line + self::$sourceCodeSpan + 1);

        $sourceExcerpt = array_slice($source, $startLine, $offset);
        
        if ($startLine > 0) {
            array_unshift($sourceExcerpt, "<?php\n", "// ...\n");
        }

        // zwraca fragment kodu rdowego wraz z wierszem pocztkowym i kocowym
        return array('source' => $sourceExcerpt,
                     'start'  => $startLine,
                     'end'    => $startLine + $offset);
    }

    // tworzy tablic zawierajc sformatowany lad zwrotny
    // uywa podkrelania skadni kodu rdowego jeeli
    // zmienna $outputFormat ma warto 'HTML'
    public static function getFormattedTrace(Exception $exception)
    {
        // inicjalizuje tablic sformatowanych elementw ladu zwrotnego
        $formattedTraces = array();

        // pobiera lad zwrotny z wyjtku
        $traces = $exception->getTrace();
        
        // inicjalizuje licznik
        $count = 1;

        // przechodzi przez kolejne elementy ladu zwrotnego
        foreach ($traces as $aTrace) {

            // pomija metod, w ktre bd zosta przeksztacony w wyjtek
            if ($aTrace['function'] != 'errorHandler') {

                // inicjalizuje tablic dla ladu
                $output = array();

                $output[] = "lad zwrotny, krok $count:";

                // generowanie nazwy klasy jeeli istnieje
                if (array_key_exists('class', $aTrace)) {
                    $output[] = array('Klasa: ', $aTrace['class']);
                }

                // generowanie typu jeeli istnieje
                if (array_key_exists('type', $aTrace)) {
                    $output[] = array('Typ: ', $aTrace['type']);
                }

                // generowanie nazwy funkcji jeeli istnieje
                if (array_key_exists('function', $aTrace)) {

                    $output[] = array('Funkcja: ', $aTrace['function']);

                    // generowanie argumentw funkcji
                    if (array_key_exists('args', $aTrace)) {
                        $output[] = array('', 'z argumentami: ' . implode(', ', $aTrace['args']));
                    }
                }

                // pobiera kodu rdowy pliku, ktry wygenerowa wyjtek
                $sourceExcerpt = self::getSourceExcerpt($aTrace['file'], $aTrace['line']);

                $output[] = 'Fragment kodu rdowego od wiersza ' . $sourceExcerpt['start']
                            . ' do wiersza ' . $sourceExcerpt['end'] . ' z pliku ' . $aTrace['file'] . ':';
        
                // wyrnia skadni dla formatu HTML
                if (self::$outputFormat == self::HTML) {
                    $output[] = array('', highlight_string(implode('', $sourceExcerpt['source']), TRUE));
                } elseif (self::$outputFormat == self::PLAIN) {
                    $output[] = implode('', $sourceExcerpt['source']);
                }

                $formattedTraces[] = $output;

                // zwiksza warto licznika
                $count++;
            }
        }

        return $formattedTraces;
    }

    // formatuje zmienne i obiekty przesane do konstruktora
    // i zachowane w $addedDebug. Uywa wyrniania skadni
    // jeeli $outputFormat ma warto 'HTML'
    public static function getFormattedDebugInfo(Exception $exception)
    {
        // inicjalizuje tablic z wygenerowan treci
        $output = array();

        // tylko klasa DebugException ma waciwo addedDebug
        if (get_class($exception) == __CLASS__) {
            
            if (count($exception->addedDebug) > 0) {
                $output[] = 'Dodatkowe informacje diagnostyczne:';
            }
            
            // przechodzi przez kad zmienn
            foreach ($exception->addedDebug as $addBug) {
                foreach ($addBug as $debugLabel => $debugVar) {
                
                    // formatuje za pomoc print_r
                    if (self::$outputFormat == self::HTML) {
                        $output[] = array($debugLabel, '<pre>' . print_r($debugVar, TRUE) . '</pre>');
                    } elseif (self::$outputFormat == self::PLAIN) {
                        $output[] = array($debugLabel, print_r($debugVar, TRUE));
                    }
                }
            }
        }
        
        return $output;
    }
    
    // przeksztaca tablic elementw do wywietlenia w tabel HTML
    // oczekiwany format:
    //         array('jaki tekst',         <- pojedyncza komrka w wierszu 1
    //               array('label', $value),   <- dwie komrki w wierszu 2
    //                                             (etykieta i warto)
    //               ...);
    public static function arrayToTable(array $contents = array(),
                                        $id = null,
                                        $caption = null,
                                        $hideByDefault = TRUE)
    {
        $html = '';

        // otwiera znacznik tablicy
        if (count($contents) > 0) {
            $html .= '<table style="width: 100%;border: 2px solid #666;display: ';
            $html .= ($hideByDefault) ? 'none' : '';
            $html .= ';"';
            $html .= ($id != null) ? " id=\"$id\"" : '';
            $html .= ">\n";
        }

        // dodaje opis
        if (!empty($caption) > 0) {
            $html .= '<caption><h2>' . htmlentities($caption) . "</h2></caption>\n";
        }

        $rowCount = 1;
        $rowColors = array('#fff', '#ccc');
        
        // przechodzi przez tablic wyjciow
        foreach ($contents as $row) {
            $html .= "<tr style=\"background: " . $rowColors[($rowCount % 2)] . ";\">\n";
            
            // dzieli tablic na etykiety i pola
            if (is_array($row) && count($row) >= 2) {
                $html .= '<td><strong>' . htmlentities($row[0]) . "</strong></td>\n"
                        . '<td>' . $row[1] . "</td>\n";
                        
            // pojedyncze cigi tekstowe wstawia do odrbnego wiersza
            } else {
                $html .= '<th colspan="2" style="text-align: left;">' . $row . "</th>\n";
            }
            
            $html .= "</tr>\n";
            
            $rowCount++;
        }

        // zamyka znacznik tabeli HTML
        if (count($contents) > 0) {
            $html .= "</table>\n";
        }
        
        return $html;
    }
    
    // spaszcza tabel na potrzeby wywietlania w trybie tekstowym
    public static function flattenArray(array $inputArray = array())
    {
       $outputArray = array();
       
       // przechodzi przez elementy tablicy
       foreach ($inputArray as $item) {
       
           if (is_array($item)) {
               // przechodzi przez ca hierarchi za pomoc rekurencji
               $outputArray = array_merge($outputArray, self::flattenArray($item));
           } else {
               array_push($outputArray, $item);
           }
       }
       
       return $outputArray;
    }
}

DebugException::init();

?>