<?php
namespace PhpMysql\CMS;                                  // Deklaracja przestrzeni nazw

class Uczestnik
{
    protected $db;                                       // Zawiera referencję do obiektu Baza

    public function __construct(Baza $db)
    {
        $this->db = $db;                                 // Dodaj referencję do obiektu Baza
    }

    // Pobierz wybranego uczestnika na podstawie id
    public function pobierz(int $id)
    {
        $sql = "SELECT id, imie, nazwisko, email, dolaczyl, zdjecie, rola
                  FROM uczestnicy
                 WHERE id = :id;";                           // SQL pobierający uczestnika
        return $this->db->wykonajSQL($sql, [$id])->fetch();  // Zwróć uczestnika
    }

    // Get details of all members
    public function pobierzWszystko(): array
    {
        $sql = "SELECT id, imie, nazwisko, dolaczyl, zdjecie, rola 
                  FROM uczestnicy;";                         // SQL pobierający wszystkich uczestników
        return $this->db->wykonajSQL($sql)->fetchAll();      // Zwróć wszystkich uczestników
    }

    // Pobierz wybranego uczestnika na podstawie e-maila
    public function pobierzIdPrzezEmail(string $email)
    {
        $sql = "SELECT id
                  FROM uczestnicy
                 WHERE email = :email;";                             // Zapytanie SQL o id użytkownika
        return $this->db->wykonajSQL($sql, [$email])->fetchColumn(); // Wykonaj SQL i zwróć id użytkownika
    }

    // Logowanie: zwraca dane uczestnika po autoryzacji lub false w przeciwnym razie
    public function logowanie(string $email, string $haslo)
    {
        $sql = "SELECT id, imie, nazwisko, dolaczyl, email, haslo, zdjecie, rola 
                  FROM uczestnicy 
                 WHERE email = :email;";                              // Pytanie SQL o dane użytkownika
        $uczestnik = $this->db->wykonajSQL($sql, [$email])->fetch();  // Wykonaj SQL
        if (!$uczestnik) {                                            // Jeśli nie znaleziono użytkownika
            return false;                                             // Zwróć false
        }                                                             // W przeciwnym razie
        $autoryzacja = password_verify($haslo, $uczestnik['haslo']);  // Czy hasła są zgodne?
        return ($autoryzacja ? $uczestnik : false);                   // Zwróć dane o użytkowniku lub false
    }

    // Pobierz liczbę uczestników
    public function policz(): int
    {
        $sql = "SELECT COUNT(id) FROM uczestnicy;";                   // Zapytanie SQL o liczbę uczestników
        return $this->db->wykonajSQL($sql)->fetchColumn();            // Wykonaj SQL i zwróć liczbę
    }

    // Utwórz nowego uczestnika
    public function utworz(array $uczestnik): bool
    {
        $uczestnik['haslo'] = password_hash($uczestnik['haslo'], PASSWORD_DEFAULT);  // Haszowanie hasła
        try {                                                          // Spróbuj dodać uczestnika
		            $sql = "INSERT INTO uczestnicy (imie, nazwisko, email, haslo) 
                    VALUES (:imie, :nazwisko, :email, :haslo);";       // SQL dodający uczestnika
            $this->db->wykonajSQL($sql, $uczestnik);                   // Wykonaj SQL
            return true;                                               // Zwróć true
        } catch (\PDOException $e) {                                   // W razie wyjątku PDOException
            if ($e->errorInfo[1] === 1062) {                           // Jeśli błąd oznacza powtórzenie danych
                return false;                                          // Zwróć false, aby zasygnalizować istnienie e-maila
            }                                                          // W przeciwnym razie
            throw $e;                                                  // Ponownie wyrzuć wyjątek
        }
    }

    // Aktualizuj uczestnika
    public function aktualizuj(array $uczestnik): bool
    {
        unset($uczestnik['dolaczyl'],  $uczestnik['zdjecie']);       // Usuń dane o dołączeniu i zdjęciu z tablicy
        try {                                                        // Spróbuj zaktualizować uczestnika
            $sql = "UPDATE uczestnicy 
                       SET imie = :imie, nazwisko = :nazwisko, email = :email, rola = :rola 
                     WHERE id = :id;";                               // SQL aktualizujący uczestnika
            $this->db->wykonajSQL($sql, $uczestnik);                 // Wykonaj SQL
            return true;                                             // Zwróć true
        } catch (\PDOException $e) {                                 // W razie wyrzucenia wyjątku PDOException
            if ($e->errorInfo[1] == 1062) {                          // Jeśli e-mail istnieje już w bazie
                return false;                                        // Zwróć false
            }                                                        // W przypadku innego błędu
            throw $e;                                                // Ponownie wyrzuć wyjątek
        }
    }

    // Prześlij zdjęcie profilowe uczestnika
    public function utworzZdjecie(int $id, string $nazwapliku, string $tymcz, string $cel): bool
    {
        if ($tymcz) {                                    // Jeśli obraz został przesłany
            $tymcz = realpath($tymcz);                   // (Na potrzeby systemu Windows)
            $obraz = new \Imagick($tymcz);               // Obiekt reprezentujący obraz
            $obraz->cropThumbnailImage(350, 350);        // Wykadruj obraz
            $zapisany = $obraz->writeImage($cel);        // Zapisz plik
            if ($zapisany == false) {                    // Jeśli zapisanie pliku się nie powiodło
                throw new \Exception('Nie udało się zapisać zdjęcia'); // Wyrzuć wyjątek
            }
        }

        $sql = "UPDATE uczestnicy 
                   SET zdjecie = :zdjecie
                 WHERE id = :id;";                                        // SQL aktualizujący zdjęcie
        $this->db->wykonajSQL($sql, ['id'=>$id, 'zdjecie'=>$nazwapliku]); // Przekaż SQL id użytkownika i nazwę pliku
        return true;                                                      // Zwróć true
    }

    // Usuń zdjęcie profilowe uczestnika
    public function usunZdjecie(array $uczestnik, string $wyslane): bool
    {
        $sciezka = $wyslane . $uczestnik['zdjecie'];      // Utwórz ścieżkę dla zdjęcia
        $usun = unlink($sciezka);                         // Usuń plik ze zdjęciem
        if ($usun === false) {                            // Wyrzuć wyjątek w razie niepowodzenia
            throw new \Exception('Nie udało się usunąć zdjęcia lub zdjęcie nie istnieje');
        }
        $sql = "UPDATE uczestnicy 
                   SET zdjecie = null
                 WHERE id = :id;";                        // SQL zmieniający zdjęcie na null
        $this->db->wykonajSQL($sql, ['id'=>$uczestnik['id']]);  // Wykonaj SQL
        return true;                                      // Zwróć true
    }

    // Aktualizuj hasło uczestnika
    public function aktualizacjaHasla(int $id, string $haslo): bool
    {
        $hash = password_hash($haslo, PASSWORD_DEFAULT);               // Utwórz hasz dla hasła
        $sql = 'UPDATE uczestnicy 
                   SET haslo = :haslo 
                 WHERE id = :id;';                                     // SQL aktualizujący hasło
        $this->db->wykonajSQL($sql, ['id' => $id, 'haslo' => $hash,]); // Wykonaj SQL
        return true;                                                   // Zwróć true
    }
}