<?php

class Publikacja
{
    protected $db;                                       // Zawiera referencję do obiektu Baza

    public function __construct(Baza $db)
    {
        $this->db = $db;                                 // Tworzy referencję do obiektu Baza
    }

    public function pobierz(int $id, bool $opublikowano = true)
    {
        $sql = "SELECT p.id, p.tytul, p.podsumowanie, p.tresc, p.utworzono, p.id_kategorii, 
                       p.id_uczestnika, p.opublikowano,
                       k.nazwa     AS kategoria,
                       CONCAT(u.imie, ' ', u.nazwisko) AS autor,
                       o.id        AS id_obrazu, 
                       o.plik      AS obraz_plik, 
                       o.alt       AS obraz_alt 
                  FROM publikacje  AS p
                  JOIN kategorie   AS k ON p.id_kategorii   = k.id
                  JOIN uczestnicy  AS u ON p.id_uczestnika  = u.id
                  LEFT JOIN obrazy AS o ON p.id_obrazu      = o.id
                 WHERE p.id = :id ";                        // Instrukcja SQL
        if ($opublikowano) {                                // Jeśli publikacja musi być widoczna
            $sql .= "AND p.opublikowano = 1 ";              // Dodaj klauzulę do SQL
        }
        return $this->db->wykonajSQL($sql, [$id])->fetch(); // Zwróć publikację
    }

    public function pobierzWszystko($opublikowano = true, $kategoria = null, $uczestnik = null, $limit = 10000): array
    {
        $argumenty['kategoria']  = $kategoria;              // Id kategorii
        $argumenty['kategoria1'] = $kategoria;              // Id kategorii
        $argumenty['uczestnik']  = $uczestnik;              // Id autora
        $argumenty['uczestnik1'] = $uczestnik;              // Id autora
        $argumenty['limit']      = $limit;                  // Maksymalna liczba wyników

        $sql = "SELECT p.id, p.tytul, p.podsumowanie, p.utworzono, p.id_kategorii, 
                       p.id_uczestnika, p.opublikowano,
                       k.nazwa    AS kategoria,
                       CONCAT(u.imie, ' ', u.nazwisko) AS autor,
                       o.plik     AS obraz_plik, 
                       o.alt      AS obraz_alt 

                FROM publikacje   AS p
                JOIN kategorie    AS k ON p.id_kategorii  = k.id
                JOIN uczestnicy   AS u ON p.id_uczestnika = u.id
                LEFT JOIN obrazy  AS o ON p.id_obrazu     = o.id 

              WHERE (p.id_kategorii  = :kategoria OR :kategoria1 is null)
                AND (p.id_uczestnika = :uczestnik OR :uczestnik1 is null) ";  // SQL pobierający podsumowania

        if ($opublikowano) {                             // Jeśli publikacja musi być widoczna
            $sql .= "AND p.opublikowano = 1 ";           // Dodaj klauzulę do SQL
        }
        $sql .= "ORDER BY p.id DESC LIMIT :limit;";      // Dodaj sortowanie i maksymalną liczbę wyników

        return $this->db->wykonajSQL($sql, $argumenty)->fetchAll();  // Zwróć dane
    }

    // WYSZUKIWANIE
    public function policzWyniki(string $term): int
    {
        $argumenty['term1'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $argumenty['term2'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $argumenty['term3'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $sql   = "SELECT COUNT(tytul)
                FROM publikacje
               WHERE tytul        LIKE :term1 
                  OR podsumowanie LIKE :term2 
                  OR tresc        LIKE :term3
                 AND opublikowano = 1;";                 // SQL zliczający dopasowania
        return $this->db->wykonajSQL($sql, $argumenty)->fetchColumn(); // Zwróć liczbę dopasowań
    }

    public function szukaj(string $term, int $pokaz = 3, int $od = 0): array
    {
        $argumenty['term1'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $argumenty['term2'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $argumenty['term3'] = '%' . $term . '%';         // Dodaj znaki wieloznaczne do szukanego terminu
        $argumenty['pokaz']  = $pokaz;                   // Liczba wyników do wyświetlenia
        $argumenty['od']  = $od;                         // Liczba wyników do pominięcia
        $sql  = "SELECT p.id, p.tytul, p.podsumowanie, p.utworzono, p.id_kategorii, p.id_uczestnika,
                     k.nazwa      AS kategoria,
                     CONCAT(u.imie, ' ', u.nazwisko) AS autor,
                     o.plik       AS obraz_plik, 
                     o.alt        AS obraz_alt

                FROM publikacje   AS p
                JOIN kategorie    AS k    ON p.id_kategorii  = k.id
                JOIN uczestnicy   AS u    ON p.id_uczestnika = u.id
                LEFT JOIN obrazy  AS o    ON p.id_obrazu     = o.id

               WHERE p.tytul        LIKE :term1 
                  OR p.podsumowanie LIKE :term2
                  OR p.tresc        LIKE :term3
                 AND p.opublikowano = 1
               ORDER BY p.id DESC
               LIMIT :pokaz 
              OFFSET :od;";                              // SQL pobierający podsumowania publikacji
        return $this->db->wykonajSQL($sql, $argumenty)->fetchAll(); // Zwróć podsumowania publikacji
    }


    // MEDOTY ADMINISTRACYJNE
    public function policz(): int
    {
        $sql = "SELECT COUNT(id) FROM publikacje;";          // SQL zliczający publikacje
        return $this->db->wykonajSQL($sql)->fetchColumn();   // Zwróć liczbę ze zbioru rezultatów
    }

    public function utworz(array $publikacja, string $tymczasowa, string $cel): bool
    {
        try {                                                // Spróbuj wstawić dane
            $this->db->beginTransaction();                   // Rozpocznij transakcję
            if ($cel) {                                      // Jeśli obraz jest prawidłowy
                // Wykadruj i zapisz plik
                $imagick = new \Imagick($tymczasowa);        // Obiekt reprezentujący obraz
                $imagick->cropThumbnailImage(1200, 700);     // Utwórz wykadrowany obraz
                $imagick->writeImage($cel);                  // Zapisz plik

                $sql = "INSERT INTO obrazy (plik, alt)
                        VALUES (:plik, :alt);";              // SQL wstawiający obraz
                $this->db->wykonajSQL($sql, [$publikacja['obraz_plik'], $publikacja['obraz_alt']]); // Dodaj obraz do tabeli
                $publikacja['id_obrazu'] = $this->db->lastInsertId(); // Zwróć id obrazu
            }
            unset($publikacja['obraz_plik'], $publikacja['obraz_alt']);
            $sql = "INSERT INTO publikacje (tytul, podsumowanie, tresc, id_kategorii, id_uczestnika,
                           id_obrazu, opublikowano)
                    VALUES (:tytul, :podsumowanie, :tresc, :id_kategorii, :id_uczestnika,
                           :id_obrazu, :opublikowano);";  // SQL dodający publikację
            $this->db->wykonajSQL($sql, $publikacja);     // Dodaj publikację
            $this->db->commit();                          // Rozpocznij transakcję
            return true;                                  // Zwróć true
        } catch (Exception $e) {                          // W razie wystąpienia wyjątku
            $this->db->rollBack();                        // Anuluj transakcję
            if (file_exists($cel)) {                      // Jeśli istnieje plik obrazu
                unlink($cel);                             // Skasuj go
            }
            if (($e instanceof PDOException) and ($e->errorInfo[1] === 1062)) { // Jeśli błąd dotyczy spójności danych
                return false;                             // Zwróć false
            } else {                                      // W razie dowolnego innego błędu
                throw $e;                                 // Ponownie wyrzuć wyjątek
            }
        }
    }

    public function aktualizuj(array $publikacja, string $tymczasowa, string $cel): bool
    {
        try {                                             // Spróbuj zaktualizować dane
            $this->db->beginTransaction();                // Rozpocznij transakcję
            if ($cel) {                                   // Jeśli przesłano obraz
                // Wykadruj i zapisz plik
                $imagick = new \Imagick($tymczasowa);     // Obiekt reprezentujący obraz
                $imagick->cropThumbnailImage(1200, 700);  // Utwórz wykadrowany obraz
                $imagick->writeImage($cel);               // Zapisz plik

                $sql = "INSERT INTO obrazy (plik, alt)
                  VALUES (:plik, :alt);";                 // SQL dodający obraz
                $this->db->wykonajSQL($sql, [$publikacja['obraz_plik'], $publikacja['obraz_alt']]);    // Dodaj obraz do tabeli
                $publikacja['id_obrazu'] = $this->db->lastInsertId();  // Dodaj id obrazu do zmiennej $publikacja
            }
            // Usuń zbędne elementy ze zmiennej $publikacja
            unset($publikacja['kategoria'], $publikacja['utworzono'], $publikacja['autor'], $publikacja['obraz_plik'], $publikacja['obraz_alt']);
            $sql = "UPDATE publikacje SET tytul = :tytul, podsumowanie = :podsumowanie, tresc = :tresc, 
                           id_kategorii = :id_kategorii, id_uczestnika = :id_uczestnika, 
                           id_obrazu = :id_obrazu, opublikowano = :opublikowano 
                     WHERE id = :id;";                            // Instrukcja SQL
            $this->db->wykonajSQL($sql, $publikacja)->rowCount(); // Zaktualizuj publikację
            $this->db->commit();                                  // Rozpocznij transakcję
            return true;                                          // Aktualizacja się powiodła
        } catch (Exception $e) {                                  // W razie wystąpienia wyjątku
            $this->db->rollBack();                                // Anuluj transakcję
            if (file_exists($cel)) {                              // Jeśli istnieje plik obrazu
                unlink($cel);                                     // Skasuj go
            }
            if (($e instanceof PDOException) and ($e->errorInfo[1] === 1062)) {  // Jeśli błąd dotyczy spójności danych
                return false;                            // Zwróć false
            } else {                                     // W razie dowolnego innego błędu
                throw $e;                                // Ponownie wyrzuć wyjątek
            }
        }
    }

    public function usun(int $id): bool
    {
        $sql = "DELETE FROM publikacje WHERE id = :id;";    // Instrukcja SQL
        $this->db->wykonajSQL($sql, [$id]);                 // Usuń publikację
        return true;                                        // Zwróć true
    }

    public function usunObraz(int $id_obrazu, string $sciezka, int $id_publikacji)
    {
        $sql = "UPDATE publikacje SET id_obrazu = null 
               WHERE id = :id_publikacji;";                 // Instrukcja SQL
        $this->db->wykonajSQL($sql, [$id_publikacji]);      // Usuń obraz z tabeli publikacje
        $sql = "DELETE FROM obrazy 
               WHERE id = :id;";                            // Instrukcja SQL
        $this->db->wykonajSQL($sql, [$id_obrazu]);          // Usuń obraz z tabeli obrazy

        if (file_exists($sciezka)) {                        // Jeśli istnieje plik obrazu
            unlink($sciezka);                               // Usuń go
        }
    }

    public function aktualizujAlt(int $id_obrazu, string $alt)
    {
        $sql = "UPDATE obrazy SET alt = :alt 
               WHERE id = :id_publikacji;";                 // Instrukcja SQL
        $this->db->wykonajSQL($sql, [$alt, $id_obrazu]);    // Zaktualizuj opis obrazu
    }
}