<?php
class ObslugaObrazuWyjatek extends Exception {};

class ObslugaObrazu
{
    public    $typyPlikow    = ['image/jpeg', 'image/png',];        // Dozwolone typy
    protected $sciezkapliku  = '';                                  // Ścieżka do pliku
    protected $nazwapliku    = '';                                  // Nazwa pliku
    protected $daneObrazu    = [];                                  // Dane obrazu
    protected $orygSzerokosc = 0;                                   // Szerokość obrazu
    protected $orygWysokosc  = 0;                                   // Wysokość obrazu
    protected $typMime       = '';                                  // Format (MIME)

    public function __construct(string $sciezkapliku, string $nazwapliku) 
    {
        $this->sciezkapliku    = $sciezkapliku;                     // Ścieżka wysłania pliku
        $this->nazwapliku      = $nazwapliku;                       // Nazwa przesłanego pliku
        $this->daneObrazu      = getimagesize($sciezkapliku);       // Pobierz dane obrazu
        $this->orygSzerokosc   = $this->daneObrazu[0];              // Szerokość obrazu
        $this->orygWysokosc    = $this->daneObrazu[1];              // Wysokość obrazu
        $this->typMime         = $this->daneObrazu['mime'];         // Format (MIME)

        if (!in_array($this->typMime, $this->typyPlikow)) {         // W razie niedozwolonego typu MIME
            throw new ObslugaObrazuWyjatek('Nieobsługiwany format pliku', 1);
        }
    }

    public function przeskalujObraz(int $nowaSzerokosc, int $nowaWysokosc, string $sciezkaDocelowa): string
    {
        if (($this->orygSzerokosc < $nowaSzerokosc)         
        or ($this->orygWysokosc < $nowaWysokosc)) {                  // Jeśli oryginał jest za mały
            throw new ObslugaObrazuWyjatek('Oryginalny obraz jest za mały', 2);
        }

        $oryg_proporcje = $this->orygSzerokosc / $this->orygWysokosc;       // Proporcje oryginału
        $nowe_proporcje  = $nowaSzerokosc / $nowaWysokosc;                  // Proporcje kadru

        // Oblicz nowy rozmiar
        if ($nowe_proporcje < $oryg_proporcje) {                            // Jeśli nowy wsp. prop. < oryg.
            $szer_zaznaczenia  = $this->orygWysokosc * $nowe_proporcje;     // Oblicz szerokość
            $wys_zaznaczenia = $this->orygWysokosc;                         // Wysokość bez zmian
            $przesuniecie_x      = ($this->orygSzerokosc - $szer_zaznaczenia) / 2;   // Oblicz przesunięcie x
            $przesuniecie_y      = 0;                                       // Przesunięcie y = 0 (górna krawędź)
        } else {                                                            // W przeciwnym razie
            $szer_zaznaczenia  = $this->orygSzerokosc;                      // Szerokość bez zmian
            $wys_zaznaczenia = $this->orygSzerokosc * $nowe_proporcje;      // Oblicz wysokość
            $przesuniecie_x      = 0;                                       // Przesunięcie x = 0 (lewa krawędź)
            $przesuniecie_y      = ($this->orygWysokosc - $wys_zaznaczenia) / 2; // Oblicz przesuniecie y
        }

        switch($this->typMime) {                                            // Jeśli format to
            case 'image/jpeg' :                                             // JPEG
                $oryg = imagecreatefromjpeg($this->sciezkapliku);           // Otwórz JPEG
                break;                                                      // Koniec instrukcji switch
            case 'image/png' :                                              // PNG
                $oryg = imagecreatefrompng($this->sciezkapliku);            // Otwórz PNG
                break;                                                      // Koniec instrukcji switch
        }

        $nowy = imagecreatetruecolor($nowaSzerokosc, $nowaWysokosc);             // Nowy pusty obraz
        imagecopyresampled($nowy, $oryg, 0, 0, $przesuniecie_x, $przesuniecie_y, $nowaSzerokosc, 
                           $nowaWysokosc, $szer_zaznaczenia, $wys_zaznaczenia);  // Kadrowanie i skalowanie

        $sciezka = $this->createFilepath($this->nazwapliku, $sciezkaDocelowa);   // Utwórz ścieżkę pliku

        // Zapisz obraz we właściwym formacie
        switch($this->typMime) {
            case 'image/gif' : $result = imagegif($nowy, $sciezka);  break;  // Zapisz GIF 
            case 'image/jpeg': $result = imagejpeg($nowy, $sciezka); break;  // Zapisz JPEG
            case 'image/png' : $result = imagepng($nowy, $sciezka);  break;  // Zapisz PNG
        }
        return $sciezka;
    }

    public function createFilepath($nazwapliku, $sciezkaDocelowa): string
    {
        $nazwa  = pathinfo($nazwapliku, PATHINFO_FILENAME);                   // Pobierz nazwę pliku
        $rozszerzenie = pathinfo($nazwapliku, PATHINFO_EXTENSION);            // Pobierz rozszerzenie
        $nazwa  = preg_replace("/[^A-z0-9]/", "-", $nazwa);                   // Oczyść nazwę pliku
        $sciezkapliku  = $sciezkaDocelowa . $nazwa . '.' . $rozszerzenie;     // Miejsce docelowe
        $i         = 0;                                                       // Licznik
        while (file_exists($sciezkapliku)) {                                  // Jeśli plik istnieje
            $i        = $i + 1;                                               // Zaktualizuj licznik
            $sciezkapliku = $sciezkaDocelowa . $nazwa . $i . '.' . $rozszerzenie; // Nowa ścieżka pliku
        }
        return $sciezkapliku;                                                 // Zwróć ścieżkę pliku
    }

    // Miejsce na inne metody, wykonujące różne operacje na obrazie
}