<?php

declare(strict_types=1);

/**
 * Tworzy i modyfikuje kalendarz wydarzeń
 *
 * PHP w wersji 7
 *
 * LICENCJA: Plik źródłowy udostępniany na licencji MIT opisanej pod adresem
 * http://www.opensource.org/licenses/mit-license.html
 *
 * @author Jason Lengstorf <jason.lengstorf@ennuidesign.com>
 * @copyright 2009 Ennui Design
 * @license http://www.opensource.org/licenses/mit-license.html
 */

class Calendar extends DB_Connect
{

    /**
     * Data dla której powstaje kalendarz
     *
     * Przechowywana w formacie RRRR-MM-DD GG:MM:SS
     *
     * @var string data używana w kalendarzu
     */
    private $_useDate;

    /**
     * Miesiąc, dla którego powstaje kalendarz
     *
     * @var int the używany miesiąc
     */
    private $_m;

    /**
     * Rok, dla którego powstaje kalendarz
     *
     * @var int używany rok
     */
    private $_y;

    /**
     * Używana liczba dni w miesiącu
     *
     * @var int liczba dni w miesiącu
     */
    private $_daysInMonth;

    /**
     * Indeks dnia tygodnia, którym zaczyna się miesiąc (0-6)
     *
     * @var int dzień tygodnia, którym zaczyna się miesiąc
     */
    private $_startDay;

    /**
     * Tworzy obiekt bazy danych i zapisuje odpowiednie dane
     *
     * Pobiera obiekt bazy danych przy tworzeniu instancji klasy
     * i zapisuje go w prywatnej właściwości $_db, o ile nie jest równy null.
     * Jeżeli obiekt ma wartość null, tworzy i zapisuje zamiast niego
     * nowy obiekt PDO.
     *
     * W metodzie są też gromadzone i zapisywane dodatkowe informacje,
     * takie jak miesiąc, dla którego powstaje kalendarz,
     * liczba jego dni, w jakim dniu tygodnia się zaczyna
     * oraz aktualny dzień.
     *
     * @param object $dbo obiekt bazy danych
     * @param string $useDate data, dla której powstaje kalendarz
     * @return void
     */
    public function __construct($dbo=NULL, $useDate=NULL)
    {
        /*
         * Wywołaj konstruktor klasy nadrzędnej, żeby sprawdzić, czy
         * obiekt bazy danych istnieje
         */
        parent::__construct($dbo);

        /*
         * Uzyskaj i zapisz dane dotyczące miesiąca
         */
        if ( isset($useDate) )
        {
             $this->_useDate = $useDate;
        }
        else
        {
             $this->_useDate = date('Y-m-d H:i:s');
        }

        /*
         * Konwersja na znacznik czasu oraz wyznaczenie miesiąca
         * i roku, dla którego powstaje kalendarz
         */
        $ts = strtotime($this->_useDate);
        $this->_m = (int)date('m', $ts);
        $this->_y = (int)date('Y', $ts);

        /*
         * Określ liczbę dni w miesiącu
         */
        $this->_daysInMonth = cal_days_in_month(
                CAL_GREGORIAN,
                $this->_m,
                $this->_y
            );

        /*
         * Określ dzień tygodnia, jakim zaczyna się miesiąc
         */
        $ts = mktime(0, 0, 0, $this->_m, 1, $this->_y);
        $this->_startDay = (int)date('w', $ts);
    }

    /**
     * Ładuje informacje o wydarzeniu (lub wydarzeniach) do tablicy
     *
     * @param int $id opcjonalny identyfikator wydarzenia do filtrowania wyników
     * @return array tablica wydarzeń z bazy danych
     */
    private function _loadEventData($id=NULL)
    {
        $sql = "SELECT
                    `event_id`, `event_title`, `event_desc`,
                    `event_start`, `event_end`
                FROM `events`";

        /*
         * Jeżeli podano identyfikator wydarzenia, dodaj klauzulę WHERE,
         * aby zwrócone zostało tylko to wydarzenie
         */
        if ( !empty($id) )
        {
            $sql .= "WHERE `event_id`=:id LIMIT 1";
        }

        /*
         * W przeciwnym razie załaduj wszystkie wydarzenia z danego miesiąca
         */
        else
        {
            /*
             * Wyznacz pierwszy i ostatni dzień miesiąca
             */
            $start_ts = mktime(0, 0, 0, $this->_m, 1, $this->_y);
            $end_ts = mktime(23, 59, 59, $this->_m+1, 0, $this->_y);
            $start_date = date('Y-m-d H:i:s', $start_ts);
            $end_date = date('Y-m-d H:i:s', $end_ts);

            /*
             * Odfiltruj wydarzenia odbywające się w
             * aktualnie wybranym miesiącu
             */
            $sql .= "WHERE `event_start`
                        BETWEEN '$start_date'
                        AND '$end_date'
                    ORDER BY `event_start`";
        }

        try
        {
            $stmt = $this->db->prepare($sql);

            /*
             * Powiąż parametr z identyfikatorem, o ile został przekazany
             */
            if ( !empty($id) )
            {
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
            }

            $stmt->execute();
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $stmt->closeCursor();

            return $results;
        }
        catch ( Exception $e )
        {
            die ( $e->getMessage() );
        }
    }

    /**
     * Składuje wszystkie wydarzenia dla miesiąca w tablicy
     *
     * @return array Informacje o wydarzeniach
     */
    private function _createEventObj()
    {
        /*
         * Pobierz tablicę wydarzeń
         */
        $arr = $this->_loadEventData();

        /*
         * Utwórz nową tablicę, a następnie pogrupuj wydarzenia
         * według dnia miesiąca, w którym się odbywają
         */
        $events = array();
        foreach ( $arr as $event )
        {
            $day = date('j', strtotime($event['event_start']));

            try
            {
                $events[$day][] = new Event($event);
            }
            catch ( Exception $e )
            {
                die ( $e->getMessage() );
            }
        }
        return $events;
    }

    /**
     * Zwraca kod HTML kalendarza i wydarzeń
     * 
     * Na podstawie informacji zawartych we właściwościach,
     * pobiera wydarzenia dla danego miesiąca, generuje
     * kalendarz i zwraca całość w postaci poprawnego kodu HTML.
     * 
     * @return string kod HTML kalendarza
     */
    public function buildCalendar()
    {
        /*
         * Wyznacz miesiąc i utwórz tablicę
         * skróconych nazw dni tygodnia do oznaczenia kolumn kalendarza
         */
        $cal_month = date('F Y', strtotime($this->_useDate));
        define('WEEKDAYS', array('Nd', 'Pn', 'Wt', 'Śr', 'Cz', 'Pt', 'So'));

        /*
         * Dodaj nagłówek w kodzie kalendarza
         */
        $html = "\n\t<h2>$cal_month</h2>";
        for ( $d=0, $labels=NULL; $d<7; ++$d )
        {
            $labels .= "\n\t\t<li>" . WEEKDAYS[$d] . "</li>";
        }
        $html .= "\n\t<ul class=\"weekdays\">"
            . $labels . "\n\t</ul>";

        /*
         * Pobierz dane wydarzeń
         */
        $events = $this->_createEventObj();

        /*
         * Generuj kod kalendarza
         */
        $html .= "\n\t<ul>"; // Początek nowej listy nieuporządkowanej
        for ( $i=1, $c=1, $t=date('j'), $m=date('m'), $y=date('Y');
                $c<=$this->_daysInMonth; ++$i )
        {
            /*
             * Dodaj klasę "fill" do pól przed
             * pierwszym dniem miesiąca
             */
            $class = $i<=$this->_startDay ? "fill" : NULL;

            /*
             * Dodaj klasę "today", jeśli bieżąca data odpowiada
             * dzisiejszej
             */
            if ( $c==$t && $m==$this->_m && $y==$this->_y )
            {
                $class = "today";
            }

            /*
             * Generuj znaczniki otwierające i zamykające element listy
             */
            $ls = sprintf("\n\t\t<li class=\"%s\">", $class);
            $le = "\n\t\t</li>";
            $event_info = NULL;
            
            /*
             * Dodaj dzień miesiąca w polu kalendarza
             */
            if ( $this->_startDay<$i && $this->_daysInMonth>=$c)
            {
                /*
                 * Formatuj dane wydarzenia
                 */
                //$event_info = NULL; // wyczyść zmienną
                if ( isset($events[$c]) )
                {
                    foreach ( $events[$c] as $event )
                    {
                        $link = '<a href="view.php?event_id='
                                . $event->id . '">' . $event->title
                                . '</a>';
                        $event_info .= "\n\t\t\t$link";
                    }
                }

                $date = sprintf("\n\t\t\t<strong>%02d</strong>",$c++);
            }
            else { $date="&nbsp;"; }

            /*
             * Jeśli bieżący dzień to sobota, zacznij nowy wiersz
             */
            $wrap = $i!=0 && $i%7==0 ? "\n\t</ul>\n\t<ul>" : NULL;

            /*
             * Generuj gotowy element listy
             */
            $html .= $ls . $date . $event_info . $le . $wrap;
        }

        /*
         * Wypełnij ostatni tydzień do końca
         */
        while ( $i%7!=1 )
        {
            $html .= "\n\t\t<li class=\"fill\">&nbsp;</li>";
            ++$i;
        }

        /*
         * Zamknij ostatnią listę nieuporządkowaną
         */
        $html .= "\n\t</ul>\n\n";

        /*
         * Zwróć wygenerowany kod
         */
        return $html;
    }

    /**
     * Wyświetla informacje o danym wydarzeniu
     * 
     * @param int $id identyfikator wydarzenia
     * @return string kod HTML do wyświetlenia informacji o wydarzeniu
     */
    public function displayEvent($id)
    {
        /*
         * Upewnij się, że został przekazany identyfikator
         */
        if ( empty($id) ) { return NULL; }

        /*
         * Upewnij się, że identyfikator jest liczbą całkowitą
         */
        $id = preg_replace('/[^0-9]/', '', $id);

        /*
         * Pobierz informacje o wydarzeniach z bazy danych
         */
        $event = $this->_loadEventById($id);

        /*
         * Generuj łańcuchy daty, czasu rozpoczęcia i czasu zakończenia
         */
        $ts = strtotime($event->start);
        $date = date('F d, Y', $ts);
        $start = date('g:ia', $ts);
        $end = date('g:ia', strtotime($event->end));

        /*
         * Generuj i zwróć kod HTML
         */
        return "<h2>$event->title</h2>"
            . "\n\t<p class=\"dates\">$date, $start&mdash;$end</p>"
            . "\n\t<p>$event->description</p>";
    }

    /**
     * Zwraca pojedynczy obiekt wydarzenia
     * 
     * @param int $id identyfikator wydarzenia
     * @return object obiekt wydarzenia
     */
    private function _loadEventById($id)
    {
        /*
         * Jeśli nie przekazano identyfikatora, zwróć wartość NULL
         */
        if ( empty($id) )
        {
            return NULL;
        }

        /*
         * Pobierz wydarzenie i zapisz je w tablicy
         */
        $event = $this->_loadEventData($id);

        /*
         * Zwróć obiekt wydarzenia
         */
        if ( isset($event[0]) )
        {
            return new Event($event[0]);
        }
        else
        {
            return NULL;
        }
    }

}

?>