 #include <apr_uri.h>
 #include <apr_fnmatch.h>
 #include <unistd.h>

 #include "commands.h"
 #include "dbg.h"
 #include "bstrlib.h"
 #include "db.h"
 #include "shell.h"

 int Command_depends(apr_pool_t * p, const char *path)
 {
     FILE *in = NULL;
     bstring line = NULL;

     in = fopen(path, "r");
     check(in != NULL, "Nie udało się otworzyć pobranych zależności: %s", path);

     for (line = bgets((bNgetc) fgetc, in, '\n');
             line != NULL;
             line = bgets((bNgetc) fgetc, in, '\n'))
     {
         btrimws(line);
         log_info("Przetwarzanie zależności: %s", bdata(line));
         int rc = Command_install(p, bdata(line), NULL, NULL, NULL);
         check(rc == 0, "Nie udało się zainstalować: %s", bdata(line));
         bdestroy(line);
     }

     fclose(in);
     return 0;

 error:
     if (line) bdestroy(line);
     if (in) fclose(in);
     return -1;
 }

 int Command_fetch(apr_pool_t * p, const char *url, int fetch_only)
 {
     apr_uri_t info = {.port = 0 };
     int rc = 0;
     const char *depends_file = NULL;
     apr_status_t rv = apr_uri_parse(p, url, &info);

     check(rv == APR_SUCCESS, "Nie udało się przetworzyć adresu URL: %s", url);

     if (apr_fnmatch(GIT_PAT, info.path, 0) == APR_SUCCESS) {
         rc = Shell_exec(GIT_SH, "URL", url, NULL);
         check(rc == 0, "Wykonanie polecenia git zakończyło się niepowodzeniem.");
     } else if (apr_fnmatch(DEPEND_PAT, info.path, 0) == APR_SUCCESS) {
         check(!fetch_only, "Nie ma sensu pobieranie pliku zależności.");

         if (info.scheme) {
             depends_file = DEPENDS_PATH;
             rc = Shell_exec(CURL_SH, "URL", url, "TARGET", depends_file,
                     NULL);
             check(rc == 0, "Wykonanie polecenia curl zakończyło się niepowodzeniem.");
         } else {
             depends_file = info.path;
         }

         // Rekurencyjne przetwarzanie listy devpkg.
         log_info("Kompilacja zgodnie z plikiem DEPENDS: %s", url);
         rv = Command_depends(p, depends_file);
         check(rv == 0, "Nie udało się przetworzyć DEPENDS: %s", url);

         // To wskazuje na brak operacji do wykonania.
         return 0;

     } else if (apr_fnmatch(TAR_GZ_PAT, info.path, 0) == APR_SUCCESS) {
         if (info.scheme) {
             rc = Shell_exec(CURL_SH,
                     "URL", url, "TARGET", TAR_GZ_SRC, NULL);
             check(rc == 0, "Nie udało się pobrać kodu źródłowego: %s", url);
         }

         rv = apr_dir_make_recursive(BUILD_DIR,
                 APR_UREAD | APR_UWRITE |
                 APR_UEXECUTE, p);
         check(rv == APR_SUCCESS, "Nie udało się utworzyć katalogu %s",
                 BUILD_DIR);

         rc = Shell_exec(TAR_SH, "FILE", TAR_GZ_SRC, NULL);
         check(rc == 0, "Nie udało się rozpakować %s", TAR_GZ_SRC);
     } else if (apr_fnmatch(TAR_BZ2_PAT, info.path, 0) == APR_SUCCESS) {
         if (info.scheme) {
             rc = Shell_exec(CURL_SH, "URL", url, "TARGET", TAR_BZ2_SRC,
                     NULL);
             check(rc == 0, "Wykonanie polecenia curl zakończyło się niepowodzeniem.");
         }

         apr_status_t rc = apr_dir_make_recursive(BUILD_DIR,
                 APR_UREAD | APR_UWRITE
                 | APR_UEXECUTE, p);

         check(rc == 0, "Nie udało się utworzyć katalogu %s", BUILD_DIR);
         rc = Shell_exec(TAR_SH, "FILE", TAR_BZ2_SRC, NULL);
         check(rc == 0, "Nie udało się rozpakować %s", TAR_BZ2_SRC);
     } else {
         sentinel("Nie wiadomo, jak obsłużyć %s", url);
     }

     // Wskazuje na konieczność rzeczywistego przeprowadzenia instalacji.
     return 1;
 error:
     return -1;
 }

 int Command_build(apr_pool_t * p, const char *url,
         const char *configure_opts, const char *make_opts,
         const char *install_opts)
 {
     int rc = 0;

     check(access(BUILD_DIR, X_OK | R_OK | W_OK) == 0,
             "Katalog kompilacji nie istnieje: %s", BUILD_DIR);

     // Rzeczywiste przeprowadzenie instalacji.
     if (access(CONFIG_SCRIPT, X_OK) == 0) {
         log_info("Istnieje skrypt konfiguracji, przetwarzam go.");
         rc = Shell_exec(CONFIGURE_SH, "OPTS", configure_opts, NULL);
         check(rc == 0, "Nie udało się przeprowadzić konfiguracji.");
     }

     rc = Shell_exec(MAKE_SH, "OPTS", make_opts, NULL);
     check(rc == 0, "Nie udało się przeprowadzić kompilacji.");

     rc = Shell_exec(INSTALL_SH,
             "TARGET", install_opts ? install_opts : "install",
             NULL);
     check(rc == 0, "Nie udało się zainstalować.");

     rc = Shell_exec(CLEANUP_SH, NULL);
     check(rc == 0, "Nie udało się przeprowadzić operacji porządkujących po kompilacji.");

     rc = DB_update(url);
     check(rc == 0, "Nie udało się dodać tego pakietu do bazy danych.");

     return 0;

 error:
     return -1;
 }

 int Command_install(apr_pool_t * p, const char *url,
         const char *configure_opts, const char *make_opts,
         const char *install_opts)
 {
     int rc = 0;
     check(Shell_exec(CLEANUP_SH, NULL) == 0,
             "Nie udało się przeprowadzić operacji porządkujących przed kompilacją.");

     rc = DB_find(url);
     check(rc != -1, "Błąd podczas sprawdzania bazy danych.");

     if (rc == 1) {
         log_info("Pakiet %s jest już zainstalowany.", url);
         return 0;
     }

     rc = Command_fetch(p, url, 0);

     if (rc == 1) {
         rc = Command_build(p, url, configure_opts, make_opts,
                 install_opts);
         check(rc == 0, "Nie udało się przeprowadzić kompilacji: %s", url);
     } else if (rc == 0) {
         // Nie ma konieczności przeprowadzenia instalacji.
         log_info("Instalacja zależności zakończyła się powodzeniem: %s", url);
     } else {
         // Wystąpił błąd
         sentinel("Instalacja zakończyła się powodzeniem: %s", url);
     }

     Shell_exec(CLEANUP_SH, NULL);
     return 0;

 error:
     Shell_exec(CLEANUP_SH, NULL);
     return -1;
 }
