 #include <stdio.h>
 #include <assert.h>
 #include <stdlib.h>
 #include <errno.h>
 #include <string.h>

 #define MAX_DATA 512
 #define MAX_ROWS 100

 struct Address {
     int id;
     int set;
     char name[MAX_DATA];
     char email[MAX_DATA];
 };

 struct Database {
     struct Address rows[MAX_ROWS];
 };

 struct Connection {
     FILE *file;
     struct Database *db;
 };

 void die(const char *message)
 {
     if (errno) {
         perror(message);
     } else {
         printf("BŁĄD: %s\n", message);
     }

     exit(1);
 }

 void Address_print(struct Address *addr)
 {
     printf("%d %s %s\n", addr->id, addr->name, addr->email);
 }

 void Database_load(struct Connection *conn)
 {
     int rc = fread(conn->db, sizeof(struct Database), 1, conn->file);
     if (rc != 1)
         die("Nie udało się wczytać bazy danych.");
 }

 struct Connection *Database_open(const char *filename, char mode)
 {
     struct Connection *conn = malloc(sizeof(struct Connection));
     if (!conn)
         die("Błąd pamięci.");

     conn->db = malloc(sizeof(struct Database));
     if (!conn->db)
         die("Błąd pamięci.");

     if (mode == 'c') {
         conn->file = fopen(filename, "w");
     } else {
         conn->file = fopen(filename, "r+");

         if (conn->file) {
             Database_load(conn);
          }
     }

     if (!conn->file)
         die("Nie udało się otworzyć pliku.");

     return conn;
 }

 void Database_close(struct Connection *conn)
 {
     if (conn) {
         if (conn->file)
             fclose(conn->file);
         if (conn->db)
             free(conn->db);
         free(conn);
     }
 }

 void Database_write(struct Connection *conn)
 {
     rewind(conn->file);

     int rc = fwrite(conn->db, sizeof(struct Database), 1, conn->file);
     if (rc != 1)
         die("Nie udało się zapisać w bazie danych.");

     rc = fflush(conn->file);
     if (rc == -1)
          die("Nie udało się opróżnić bazy danych.");
 }

 void Database_create(struct Connection *conn)
 {
     int i = 0;

     for (i = 0; i < MAX_ROWS; i++) {
         // Utworzenie prototypu w celu jego inicjalizacji.
         struct Address addr = {.id = i,.set = 0 };
         // Przypisanie danych prototypowi.
         conn->db->rows[i] = addr;
     }
 }

 void Database_set(struct Connection *conn, int id, const char *name,
         const char *email)
 {
     struct Address *addr = &conn->db->rows[id];
     if (addr->set)
         die("Dane już istnieją, najpierw je usuń.");

     addr->set = 1;
     // OSTRZEŻENIE: Błąd, przeczytaj podrozdział "Jak to zepsuć?", aby dowiedzieć się, jak usunąć ten błąd.
     char *res = strncpy(addr->name, name, MAX_DATA);
     // Zademonstrowanie błędu wywołania strncpy().
     if (!res)
         die("Nie udało się skopiować imienia.");

     res = strncpy(addr->email, email, MAX_DATA);
     if (!res)
         die("Nie udało się skopiować adresu e-mail.");
 }

 void Database_get(struct Connection *conn, int id)
 {
     struct Address *addr = &conn->db->rows[id];

     if (addr->set) {
         Address_print(addr);
     } else {
         die("Identyfikator nie został zdefiniowany.");
     }
 }

 void Database_delete(struct Connection *conn, int id)
 {
     struct Address addr = {.id = id,.set = 0 };
     conn->db->rows[id] = addr;
 }

 void Database_list(struct Connection *conn)
 {
     int i = 0;
     struct Database *db = conn->db;

     for (i = 0; i < MAX_ROWS; i++) {
         struct Address *cur = &db->rows[i];

         if (cur->set) {
             Address_print(cur);
         }
     }
 }

 int main(int argc, char *argv[])
 {
     if (argc < 3)
         die("UŻYCIE: ex17 <baza> <akcja> [parametry akcji]");

     char *filename = argv[1];
     char action = argv[2][0];
     struct Connection *conn = Database_open(filename, action);
     int id = 0;

     if (argc > 3) id = atoi(argv[3]);
     if (id >= MAX_ROWS) die("Nie ma aż tylu rekordów.");

     switch (action) {
         case 'c':
             Database_create(conn);
             Database_write(conn);
             break;

         case 'g':
             if (argc != 4)
                 die("Trzeba podać identyfikator.");

             Database_get(conn, id);
             break;

         case 's':
             if (argc != 6)
                 die("Trzeba podać identyfikator, imię i adres e-mail.");

             Database_set(conn, id, argv[4], argv[5]);
             Database_write(conn);
             break;

         case 'd':
             if (argc != 4)
             die("Aby usunąć rekord, trzeba podać jego identyfikator.");

             Database_delete(conn, id);
             Database_write(conn);
             break;

         case 'l':
             Database_list(conn);
             break;
         default:
             die("Nieprawidłowa akcja: c=utwórz, g=pobierz, s=ustaw, d=usuń, l=wyświetl");
     }

     Database_close(conn);

     return 0;
 }
