-- Ten skrypt uruchomi si w Oracle Database 11g lub nowszej.

-- Ten skrypt:
--   1. Tworzy uytkownika xml_user
--   2. Tworzy tebele i wypenia je danymi
--   3. Tworzy kod PL/SQL 

-- Usunicie uytkownika xml_user jeeli istnieje.
DROP USER xml_user CASCADE;

-- To polecenie moe zwrci bd jeli uytkownik nie istnieje; nie naley si tym przejmowa.
CREATE USER xml_user IDENTIFIED BY xml_password;

-- nadanie uytkownikowi uprawnie do czenia si i tworzenia obiektw bazy danych
GRANT connect, resource, create any directory TO xml_user;

-- przydzielenie uytkownikowi miejsca w przestrzeni tabel users
ALTER USER xml_user QUOTA 10M ON users;

-- poczenie jako uytkownik xml_user
CONNECT xml_user/xml_password;

-- tworzenie typw 
CREATE TYPE t_product AS OBJECT (
  product_id INTEGER,
  name VARCHAR2(45),
  quantity INTEGER
);
/

CREATE TYPE t_nested_table_product AS TABLE OF t_product;
/

-- tworzenie tabel 
CREATE TABLE purchase_order (
  purchase_order_id INTEGER CONSTRAINT purchase_order_pk PRIMARY KEY,
  customer_order_id INTEGER,
  order_date DATE,
  customer_name VARCHAR2(25),
  street VARCHAR2(15),
  city VARCHAR2(15),
  state VARCHAR2(3),
  zip VARCHAR2(5),
  phone_number VARCHAR2(12),
  products t_nested_table_product,
  xml_purchase_order XMLType
)
NESTED TABLE products
STORE AS nested_products;

CREATE OR REPLACE DIRECTORY XML_FILES_DIR AS 'C:\xml_files';

INSERT INTO purchase_order (
  purchase_order_id,
  xml_purchase_order
) VALUES (
  1,
  XMLType(
    BFILENAME('XML_FILES_DIR', 'purchase_order.xml'),
    NLS_CHARSET_ID('AL32UTF8')
  )
); 
COMMIT;

CREATE PROCEDURE update_purchase_order(
  p_purchase_order_id IN purchase_order.purchase_order_id%TYPE
) AS
  v_count INTEGER := 1;

  -- deklaracja tabeli zagniedonej, w ktrej bd skadowane produkty
  v_nested_table_products t_nested_table_product :=
    t_nested_table_product();

  -- deklaracja typu reprezentujcego rekord produktu
  TYPE t_product_record IS RECORD (
    product_id INTEGER,
    name VARCHAR2(45),
    quantity INTEGER
  );

  -- deklaracja typu REF CURSOR wskazujcego na rekordy produktw
  TYPE t_product_cursor IS REF CURSOR RETURN t_product_record;

  -- deklaracja kursora
  v_product_cursor t_product_cursor;

  -- deklaracja zmiennej przechowujcej rekord produktu
  v_product t_product_record;
BEGIN
  -- otwiera v_product_cursor do odczytania product_id, name, and quantity 
  -- kadego produktu skadowanego w XML z kolumny xml_purchase_order 
  -- tabeli purchase_order table
  OPEN v_product_cursor FOR
  SELECT
    EXTRACTVALUE(product.COLUMN_VALUE, '/product/product_id')
      AS product_id,
    EXTRACTVALUE(product.COLUMN_VALUE, '/product/name') AS name,
    EXTRACTVALUE(product.COLUMN_VALUE, '/product/quantity') AS quantity
  FROM TABLE(
    SELECT
      XMLSEQUENCE(EXTRACT(xml_purchase_order, '/purchase_order//product'))
    FROM purchase_order
    WHERE purchase_order_id = p_purchase_order_id
  ) product;

  -- ptla przez zawarto v_product_cursor
  LOOP
    -- pobiera rekordy produktw z v_product_cursor 
    -- przerywa, gdy nie ma wicej rekordw
    FETCH v_product_cursor INTO v_product;
    EXIT WHEN v_product_cursor%NOTFOUND;

    -- rozszerza tabel v_nested_table_products, 
    -- aby mona byo w niej zapisa produkt
    v_nested_table_products.EXTEND;

    -- tworzy nowy produkt i zapisuje go w in v_nested_table_products
    v_nested_table_products(v_count) :=
      t_product(v_product.product_id, v_product.name, v_product.quantity);

    -- wywietla nowy produkt zapisany w v_nested_table_products
    DBMS_OUTPUT.PUT_LINE('product_id = ' ||
      v_nested_table_products(v_count).product_id);
    DBMS_OUTPUT.PUT_LINE('name = ' ||
      v_nested_table_products(v_count).name);
    DBMS_OUTPUT.PUT_LINE('quantity = ' ||
      v_nested_table_products(v_count).quantity);

    -- zwiksza v_count przygotowujc kolejn iteracj ptli
    v_count := v_count + 1;
  END LOOP;

  -- zamyka v_product_cursor
  CLOSE v_product_cursor;

  -- modyfikuje tabel purchase_order uywajc 
  -- wartoci wydobytych z XML skadowanego w kolumnie 
  -- xml_purchase_order (zagniedona tabela products 
  -- jest ustawiana na obiekt v_nested_table_products, 
  -- w ktrym znajduj si dane z poprzedniej ptli)
  UPDATE purchase_order
  SET
    customer_order_id =
      EXTRACTVALUE(xml_purchase_order,
        '/purchase_order/customer_order_id'),
    order_date =
      TO_DATE(EXTRACTVALUE(xml_purchase_order,
        '/purchase_order/order_date'), 'YYYY-MM-DD'),
    customer_name =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/customer_name'),
    street =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/street'),
    city =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/city'),
    state =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/state'),
    zip =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/zip'),
    phone_number =
      EXTRACTVALUE(xml_purchase_order, '/purchase_order/phone_number'),
    products = v_nested_table_products
  WHERE purchase_order_id = p_purchase_order_id;

  -- zatwierdza transakcj
  COMMIT;
END update_purchase_order;
/
