/* ================================================================
||   Program: create_system_triggers.sql
||   Data:       2013-07-25
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   12
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   Ten skrypt ilustruje wyzwalacze systemowe.
|| ================================================================*/

SET ECHO OFF
SET FEEDBACK ON
SET NULL '<Null>'
SET PAGESIZE 999
SET SERVEROUTPUT ON

DECLARE

  -- Definicja wyjątku.
  wrong_schema EXCEPTION;
  PRAGMA EXCEPTION_INIT(wrong_schema,-20001);

  -- Definicja zwracanej zmiennej.
  retval VARCHAR2(1 CHAR);

 /*
  || Definicja kursora potrzebna do określenia, czy bieżący użytkownik to
  || SYSTEM lub inny użytkownik z uprawnieniami administratora.
  */
  CURSOR privs IS
    SELECT   DISTINCT null
    FROM     user_role_privs
    WHERE    username = 'SYSTEM'
    OR       granted_role = 'DBA';

BEGIN

  -- Otwieranie kursora i warunkowe usuwanie tabeli z odpowiedniego schematu.
  OPEN privs;
  LOOP
    FETCH privs INTO retval;
    IF privs%NOTFOUND THEN
      RAISE wrong_schema;
    ELSE
      FOR i IN (SELECT null
                FROM   user_tables
                WHERE  table_name = 'CONNECTION_LOG') LOOP
        EXECUTE IMMEDIATE 'DROP TABLE connection_log CASCADE CONSTRAINTS';
      END LOOP;
      FOR i IN (SELECT null
                FROM   user_sequences
                WHERE  sequence_name = 'CONNECTION_LOG_S1') LOOP
        EXECUTE IMMEDIATE 'DROP SEQUENCE connection_log_s1';
      END LOOP;
    END IF; 
    EXIT;
  END LOOP;
  CLOSE privs;

EXCEPTION

  -- Obsługa zdefiniowanego wyjątku.
  WHEN wrong_schema THEN
    DBMS_OUTPUT.PUT_LINE('Ten skrypt wymaga konta SYSTEM (a ty używasz '
    ||                   'konta <'||user||'>) lub konta z uprawnieniami '
    ||                   'administratora.');

  -- Obsługa ogólnego wyjątku.
  WHEN others THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    RETURN;

END;
/

-- Tworzenie kontrolnej tabeli connection_log.
CREATE TABLE connection_log 
( event_id           NUMBER(10)
, event_user_name    VARCHAR2(30) CONSTRAINT log_event_nn1 NOT NULL
, event_type         VARCHAR2(14) CONSTRAINT log_event_nn2 NOT NULL
, event_date         DATE         CONSTRAINT log_event_nn3 NOT NULL
, CONSTRAINT connection_log_p1    PRIMARY KEY (event_id));

-- Tworzenie sekwencji rozpoczynającej się od 1 i zwiększanej o 1.
CREATE SEQUENCE connection_log_s1;

-- Tworzenie wyzwalacza automatyzującego generowanie sztucznych kluczy głównych.
CREATE OR REPLACE TRIGGER connection_log_t1
  BEFORE INSERT ON connection_log
  FOR EACH ROW
  WHEN (new.event_id IS NULL)
BEGIN
  SELECT   connection_log_s1.nextval
  INTO     :new.event_id
  FROM     dual;
END;
/

-- Przyznawanie uprawnień do dostępu kontu PHP.
GRANT SELECT ON connection_log TO PHP;

-- Definiowanie pakietu z procedurami connecting i disconnecting.
CREATE OR REPLACE PACKAGE user_connection AS

  PROCEDURE Connecting
  (user_name       IN VARCHAR2);

  PROCEDURE Disconnecting
  (user_name       IN VARCHAR2);

END user_connection;
/

-- Definiowanie ciała pakietu z implementacją procedur.
CREATE OR REPLACE PACKAGE BODY user_connection AS

  PROCEDURE connecting
  (user_name       IN VARCHAR2) IS
  BEGIN
    INSERT INTO connection_log
    (event_user_name, event_type, event_date)
    VALUES
    (user_name,'CONNECT',SYSDATE);
  END connecting;
  
  PROCEDURE disconnecting
  (user_name       IN VARCHAR2) IS
  BEGIN
    INSERT INTO connection_log
    (event_user_name, event_type, event_date)
    VALUES
    (user_name,'DISCONNECT',SYSDATE);
  END disconnecting;
  
END user_connection;
/

-- Definicja systemowego wyzwalacza dla procesu logowania.
CREATE OR REPLACE TRIGGER connecting_trigger
  AFTER LOGON ON DATABASE
BEGIN
  user_connection.connecting(sys.login_user);
END;
/

-- Definicja systemowego wyzwalacza dla procesu wylogowywania.
CREATE OR REPLACE TRIGGER disconnecting_trigger
  BEFORE LOGOFF ON DATABASE
BEGIN
  user_connection.disconnecting(sys.login_user);
END;
/