/* ================================================================
||   Program:    expanding_view.sql
||   Data:       2013-12-02
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   2
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   Ten skrypt tworzy obiekty używane w kompletnym przykładzie ilustrującym
||   używanie nowej procedury EXPAND_SQL_TEXT z pakietu DBMS_UTILITY.
|| ================================================================*/

-- Tworzenie używanego dalej widoku na podstawie złączenia dwóch tabel.
CREATE OR REPLACE VIEW group_view AS
  SELECT   cr.full_name
  FROM     member m INNER JOIN current_rental cr
  ON       m.account_number = cr.account_number
  WHERE    m.member_type =
            (SELECT   common_lookup_id
             FROM     common_lookup
             WHERE    common_lookup_type = 'GROUP');

-- Opisywanie widoku GROUP_VIEW.
DESC group_view

-- Przekształcanie kolumny typu LONG na typ CLOB.
CREATE OR REPLACE FUNCTION long_to_clob
( pv_view_name      VARCHAR2
, pv_column_length  INTEGER )
RETURN CLOB AS

  /* Deklarowanie zmiennych lokalnych. */
  lv_cursor    INTEGER := dbms_sql.open_cursor;
  lv_feedback  INTEGER;         -- Potwierdzenie wykonania dynamicznej instrukcji
  lv_length    INTEGER;         -- Długość łańcucha znaków
  lv_return    CLOB;            -- Dane wyjściowe funkcji
  lv_stmt      VARCHAR2(2000);  -- Dynamiczna instrukcja SQL-a
  lv_string    VARCHAR2(32760); -- Maksymalna długość typu LONG

BEGIN
  /* Tworzenie dynamicznej instrukcji. */
  lv_stmt := 'SELECT text'||CHR(10)
          || 'FROM user_views'||CHR(10)
          || 'WHERE view_name = '''||pv_view_name||'''';

  /* Parsowanie instrukcji i definiowanie kolumny typu LONG. */
  dbms_sql.parse(lv_cursor, lv_stmt, dbms_sql.native);
  dbms_sql.define_column_long(lv_cursor,1);

  /* Dane są przetwarzane tylko wtedy, gdy są dostępne. */
  IF dbms_sql.execute_and_fetch(lv_cursor) = 1 THEN
    dbms_sql.column_value_long(
        lv_cursor
      , 1
      , pv_column_length
      , 0
      , lv_string
      , lv_length);
  END IF;

  /* Sprawdzanie, czy kursor jest otwarty. */
  IF dbms_sql.is_open(lv_cursor) THEN
    dbms_sql.close_cursor(lv_cursor);
  END IF;

  /* Tworzenie lokalnego tymczasowego obiektu CLOB w pamięci:
     - Zwraca utworzony element lv_return_result.
     - Unieważnia wersję z pamięci podręcznej.
     - Czas wykonywania jest ustawiony na 12 (jest to wartość zmiennej 
       dbms_lob.call z poziomu pakietu); wartość domyślna to 10. */
  dbms_lob.createtemporary(lv_return, FALSE, dbms_lob.call);

  /* Zapisywanie danych typu Long w pustym tymczasowym obiekcie typu CLOB. */
  dbms_lob.write(lv_return, pv_column_length, 1, lv_string);

  RETURN lv_return;
END long_to_clob;
/

LIST

-- Opis funkcji LONG_TO_CLOB.
DESC long_to_clob

-- Tworzenie funkcji używającej procedury dbms_utility.expand_sql_text.
CREATE OR REPLACE FUNCTION expand_view 
( pv_view_name  VARCHAR2 ) RETURN CLOB IS

  /* Deklaracje zmiennych na widoki. */
  lv_input_view   CLOB;
  lv_output_view  CLOB;

  /* Deklaracja docelowej zmiennej (potrzeba z uwagi na ograniczenia instrukcji SELECT-INTO). */
  lv_long_view  LONG;

  /* Deklaracja kursora dynamicznego. */
  CURSOR c (cv_view_name VARCHAR2) IS
    SELECT   text
    FROM     user_views
    WHERE    view_name = cv_view_name;
 
BEGIN
  /* Otwieranie kursora, pobieranie danych (tekstu widoku) i zamykanie kursora. */
  OPEN c(pv_view_name);
  FETCH c INTO lv_long_view;
  CLOSE c;

  /* Przekształcanie wartości typu LONG na typ CLOB. */
  lv_input_view := long_to_clob(pv_view_name, LENGTH(lv_long_view));

  /* Przekazywanie tekstu widoku i pobieranie kompletnego tekstu. */
  dbms_utility.expand_sql_text(lv_input_view, lv_output_view);

  /* Zwracanie wyjściowej wartości typu CLOB uzupełnionej dwoma spacjami. */
  RETURN '  '||lv_output_view;
END;
/

-- Opisywanie funkcji EXPAND_VIEW.
DESC expand_view

-- Pobieranie danych z funkcji RETURN_VIEW_TEXT.
SELECT return_view_text('GROUP_VIEW') AS "Zawartość widoku"
FROM   dual;

-- Określanie długości kolumny.
COLUMN querytext FORMAT A30 HEADING "Tekst zapytania"

-- Testowanie funkcji expand_view.
SELECT SUBSTR(expand_view('GROUP_VIEW'),1,INSTR(expand_view('GROUP_VIEW'),'"',1,4)) AS "querytext"
,      LENGTH(expand_view('GROUP_VIEW')) AS "Długość zapytania"
FROM   dual;

-- Przekształcanie kolumny typu long na typ CLOB.
CREATE OR REPLACE FUNCTION long_to_varchar2
( pv_view_name      VARCHAR2
, pv_column_length  INTEGER )
RETURN VARCHAR2 AS

  /* Deklarowanie zmiennych lokalnych. */
  lv_cursor    INTEGER := dbms_sql.open_cursor;
  lv_feedback  INTEGER;          -- Potwierdzenie wykonania instrukcji dynamicznej
  lv_length    INTEGER;          -- Długość łańcucha znaków
  lv_return    VARCHAR2(32767);  -- Dane wyjściowe funkcji
  lv_stmt      VARCHAR2(2000);   -- Dynamiczna instrukcja SQL-a
  lv_string    VARCHAR2(32760);  -- Maksymalna długość typu LONG

BEGIN
  /* Tworzenie dynamicznej instrukcji. */
  lv_stmt := 'SELECT text'||CHR(10)
          || 'FROM user_views'||CHR(10)
          || 'WHERE view_name = '''||pv_view_name||'''';

  /* Parsowanie instrukcji i definiowanie kolumny typu long. */
  dbms_sql.parse(lv_cursor, lv_stmt, dbms_sql.native);
  dbms_sql.define_column_long(lv_cursor,1);

  /* Dane są przetwarzane tylko wtedy, gdy zostały zwrócone. */
  IF dbms_sql.execute_and_fetch(lv_cursor) = 1 THEN
    dbms_sql.column_value_long(
        lv_cursor
      , 1
      , pv_column_length
      , 0
      , lv_string
      , lv_length);
  END IF;

  /* Sprawdzanie, czy kursor jest otwarty. */
  IF dbms_sql.is_open(lv_cursor) THEN
    dbms_sql.close_cursor(lv_cursor);
  END IF;

  /* Ustawianie maksymalnej długości dla łańcucha znaków. */
  lv_return := lv_string;

  RETURN lv_return;
END long_to_varchar2;
/

-- Opisywanie funkcji LONG_TO_VARCHAR2.
DESC long_to_varchar2

-- Tworzenie funkcji używającej procedury dbms_utility.expand_sql_text.
CREATE OR REPLACE FUNCTION expand_view 
( pv_view_name  VARCHAR2 ) RETURN CLOB IS

  /* Deklaracje zmiennych na widoki. */
  lv_input_view   CLOB;
  lv_output_view  CLOB;

  /* Deklaracja docelowej zmiennej (potrzebnej z powodu ograniczeń instrukcji SELECT-INTO). */
  lv_long_view  LONG;

  /* Deklaracja kursora dynamicznego. */
  CURSOR c (cv_view_name VARCHAR2) IS
    SELECT   text
    FROM     user_views
    WHERE    view_name = cv_view_name;
 
BEGIN
  /* Otwieranie kursora, pobieranie danych (tekstu widoku) i zamykanie kursora. */
  OPEN c(pv_view_name);
  FETCH c INTO lv_long_view;
  CLOSE c;

  /* Przekształcanie wartości typu LONG na typ CLOB. */
  lv_input_view := long_to_clob(pv_view_name, LENGTH(lv_long_view));

  /* Przekazywanie tekstu widoku i pobieranie kompletnego tekstu. */
  dbms_utility.expand_sql_text(lv_input_view, lv_output_view);

  /* Zwracanie wyjściowej wartości typu CLOB. */
  RETURN lv_output_view;
END;
/

-- Opisywanie funkcji EXPAND_VIEW.
DESC expand_view

-- Tworzenie funkcji zwracającej tekst widoku.
CREATE OR REPLACE FUNCTION return_view_text 
( pv_view_name  VARCHAR2 ) RETURN VARCHAR2 IS

  /* Deklaracja zmiennej docelowej (potrzebnej z powodu ograniczeń instrukcji SELECT-INTO). */
  lv_long_view  LONG;

  /* Deklaracja kursora dynamicznego. */
  CURSOR c (cv_view_name VARCHAR2) IS
    SELECT   text
    FROM     user_views
    WHERE    view_name = cv_view_name;
 
BEGIN
  /* Otwieranie kursora, pobieranie tekstu widoku i zamykanie kursora. */
  OPEN c(pv_view_name);
  FETCH c INTO lv_long_view;
  CLOSE c;

  /* Zwracanie wyjściowej wartości typu CLOB uzupełnionej dwoma spacjami. */
  RETURN '  '||long_to_varchar2(pv_view_name, LENGTH(lv_long_view));
END;
/

-- Opisywanie funkcji RETURN_VIEW_TEXT.
DESC return_view_text

-- Query the RETURN_VIEW_TEXT function.
SELECT return_view_text('GROUP_VIEW') AS "Zawartość widoku"
FROM   dual;

-- Ustawianie długości kolumny.
COLUMN querytext FORMAT A30 HEADING "Tekst zapytania"

-- Testowanie funkcji expand_view.
SELECT SUBSTR(expand_view('GROUP_VIEW'),1,INSTR(expand_view('GROUP_VIEW'),'"',1,4)) AS "querytext"
,      LENGTH(expand_view('GROUP_VIEW')) AS "Długość zapytania"
FROM   dual;
