//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <string>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_0,
    POSITION_1,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablicy
// wierzchokw - wsprzdnymi wierzchokw krawdzi obiektu
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw krawdzi obiektu
//////////////////////////////////////////////////////////////////////
GLfloat position [4*6] =
{
    -1.5f, -1.1f, 0.0f, 1.0f,
    1.5f, -1.1f, 0.0f, 1.0f,
    1.5f, -1.1f, 0.0f, 1.0f,
    0.0f, 1.5f, 0.0f, 1.0f,
    0.0f, 1.5f, 0.0f, 1.0f,
    -1.5f, -1.1f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw zapytania asynchronicznego
//////////////////////////////////////////////////////////////////////
enum
{
    TRANSFORM_FEEDBACK,
    PRIMITIVES_GENERATED,
    QUERY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektu zapytania asynchronicznego
//////////////////////////////////////////////////////////////////////
GLuint query[QUERY_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu transformacji sprzonych zwrotnie
//////////////////////////////////////////////////////////////////////
GLuint transformFeedbacks;

//////////////////////////////////////////////////////////////////////
// maksymalna liczba danych wierzchokw przechowywanych w obiektach bufora
// 49152=6*4*4*4*4*4*4*4*4, gdzie 6 to pocztkowa liczba wierzchokw
//////////////////////////////////////////////////////////////////////
const int MAX_VERTEX = 2*49152*sizeof( GLfloat );

//////////////////////////////////////////////////////////////////////
// liczba iteracji pomniejszona o 1
//////////////////////////////////////////////////////////////////////
int iterationCount = 0;

//////////////////////////////////////////////////////////////////////
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru
    glClear( GL_COLOR_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // pocztkowa liczba wywietlanych wierzchokw
    GLint vertexCount = 6;

    // numer bufora z wejciowymi atrybutami wierzchokw
    int currentBuf = POSITION_0;

    // zaadowanie pocztkowych danych atrybutw wierzchokw
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[currentBuf] );
    glBufferSubData( GL_ARRAY_BUFFER, 0, sizeof( position ), position );

    // wyczenie rasteryzacji
    glEnable( GL_RASTERIZER_DISCARD );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // rekurencyjne rysowanie fraktala
    for( int i = 0; i < iterationCount; i++ )
    {
        // format atrybutw wierzchoka
        glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[currentBuf] );
        glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

        // wczenie tablic wierzchokw
        glEnableVertexAttribArray( POSITION );

        // bufor do zapisu danych zmiennych wyjciowych shadera geometrii
        glBindBufferBase( GL_TRANSFORM_FEEDBACK_BUFFER, 0, vertexBuffer[POSITION_1 - currentBuf] );

        // wczenie zapytania asynchronicznego dotyczcego liczby zapisanych prymityww
        glBeginQuery( GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN, query[TRANSFORM_FEEDBACK] );

        // wczenie trybu transformacji sprzonych zwrotnie
        glBeginTransformFeedback( GL_LINES );

        // narysowanie danych zawartych w tablicach wierzchokw
        glDrawArrays( GL_LINES, 0, vertexCount );

        // wyczenie trybu transformacji sprzonych zwrotnie
        glEndTransformFeedback();

        // wyczenie zapytania asynchronicznego
        glEndQuery( GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN );

        // sprawdzenie, czy dostpne s wyniki zapytania asynchronicznego
        GLint available;
        do
        {
            glGetQueryObjectiv( query[TRANSFORM_FEEDBACK], GL_QUERY_RESULT_AVAILABLE, &available );
        }
        while( !available );

        // pobranie liczby zapisanych prymityww
        glGetQueryObjectiv( query[TRANSFORM_FEEDBACK], GL_QUERY_RESULT, &vertexCount );

        // liczba wierzchokw rwna dwukrotnoci liczby prymityww
        vertexCount *= 2;

        // zamiana buforw
        currentBuf = POSITION_1 - currentBuf;
    }

    // wczenie rasteryzacji
    glDisable( GL_RASTERIZER_DISCARD );

    // format atrybutw wierzchoka
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[currentBuf] );
    glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // wczenie zapytania asynchronicznego dotyczcego liczby przetworzonych prymityww
    glBeginQuery( GL_PRIMITIVES_GENERATED, query[PRIMITIVES_GENERATED] );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_LINES, 0, vertexCount );

    // wczenie zapytania asynchronicznego
    glEndQuery( GL_PRIMITIVES_GENERATED );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // sprawdzenie, czy dostpne s wyniki zapytania asynchronicznego
    GLint available;
    do
    {
        glGetQueryObjectiv( query[PRIMITIVES_GENERATED], GL_QUERY_RESULT_AVAILABLE, &available );
    }
    while( !available );

    // pobranie liczby przetworzonych prymityww
    GLint primitiveCount;
    glGetQueryObjectiv( query[PRIMITIVES_GENERATED], GL_QUERY_RESULT, &primitiveCount );

    // wypisanie informacji o liczbie przetworzonych prymityww
    std::ostringstream txt;
    txt << "liczba przetworzonych prymityww: " << primitiveCount;
    DrawText8x16( 3, 3, txt.str() );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "krzywa_kocha_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_GEOMETRY_SHADER, "krzywa_kocha_gs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "krzywa_kocha_fs.glsl" ) );

    // wyjciowe zmienne shadera geometrii zapisywane w obiekcie bufora
    const GLchar *varyings[1] = { "outPosition" };
    glTransformFeedbackVaryings( program, 1, varyings, GL_SEPARATE_ATTRIBS );

    // konsolidacja programu
    LinkValidateProgram( program );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektw bufora wierzchokw (VBO)
    glGenBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_0] );
    glBufferData( GL_ARRAY_BUFFER, MAX_VERTEX, NULL, GL_DYNAMIC_COPY );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_1] );
    glBufferData( GL_ARRAY_BUFFER, MAX_VERTEX, NULL, GL_DYNAMIC_COPY );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu transformacji sprzonych zwrotnie
    glGenTransformFeedbacks( 1, &transformFeedbacks );

    // wczenie obiektu transformacji sprzonych zwrotnie
    glBindTransformFeedback( GL_TRANSFORM_FEEDBACK, transformFeedbacks );

    // utworzenie identyfikatora obiektu zapytania asynchronicznego
    glGenQueries( QUERY_SIZE, query );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteTransformFeedbacks( 1, &transformFeedbacks );
    glDeleteQueries( QUERY_SIZE, query );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
