//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////
// na podstawie: ModelAndTextureLoader.cpp i ModelAndTextureLoader.hpp
// AMD FirePro Technology SDK
// http://developer.amd.com/tools-and-sdks/graphics-development/firepro-sdk/
// AMD-FirePro-SDK-v0.9.2-Windows.exe
//////////////////////////////////////////////////////////////////////

#ifndef __MODELS__H__
#define __MODELS__H__

#include <GL/glew.h>
#include <map>
#include <vector>
#include <glm/glm.hpp>
#include <assimp/cimport.h>
#include <assimp/scene.h>

//////////////////////////////////////////////////////////////////////
// klasa obsugujca odczyt modelu 3D z pliku
//////////////////////////////////////////////////////////////////////
class ModelAndTextureLoader
{
public:
    // konstruktor
    ModelAndTextureLoader():
        assimpScene( NULL ), totalMesh( 0 ), size( 0.0f ), opacity( false ) {}

    // destruktor
    ~ModelAndTextureLoader() { Delete(); }

    // zaadowanie obiektu z pliku
    bool Load( const char *fileName );

    // rendering caej siatki
    void MeshRendering()
    {
        unsigned int iMesh = 0;
        bool uniforms[4];
        IsUniforms( uniforms );
        MeshRendering( assimpScene -> mRootNode, iMesh, uniforms );
    }

    // rendering caej siatki z obsug (p)przezroczystoci
    void MeshRenderingOpacity( bool opacity )
    {
        unsigned int iMesh = 0;
        bool uniforms[4];
        IsUniforms( uniforms );
        MeshRenderingOpacity( opacity, assimpScene -> mRootNode, iMesh, uniforms );
    }

    // wskanik na scen
    const aiScene *GetAssimpScene() { return assimpScene; };

    // pobranie rozmiaru modelu
    float GetSize() { return size; }

    // pobranie wsprzdnych rodka modelu
    const glm::vec3 &GetCenter() { return center; }

    // pobranie znacznika (p)przezroczystych materiaw
    bool GetOpacity() { return opacity; }

private:

    // numery atrybutw wierzchokw
    enum
    {
        POSITION,   // wsprzdne wierzchokw
        NORMAL,     // wsprzdne wektorw normalnych
        TEX_COORD,  // wsprzdne tekstury
        TANGENT,    // wsprzdne wektorw stycznych
        BITANGENT   // wsprzdne wektorw bistycznych
    };

    // rendering siatek z wybranego wza
    void MeshRendering( const aiNode *nd, unsigned int &currentMesh, bool uniforms[4] );

    // rendering siatek z wybranego wza z obsug (p)przezroczystoci
    void MeshRenderingOpacity( bool opacity, const aiNode *nd, unsigned int &currentMesh, bool uniforms[4] );

    // obliczenie cznej liczbie siatek w wybranym wle
    void CountTotalMesh( const aiNode *nd );

    // zaadowanie danych siatek z wybranego wza
    void MeshLoading( const aiNode *nd, unsigned int &currentMesh );

    // usunicie danych siatek z wybranego wza
    void MeshDelete( const aiNode *nd, unsigned int &currentMesh );

    // usunicie wszystkich elementw sceny
    void Delete();

    // bryy otaczajce dla poszczeglnych wzw modelu
    void GetBoundingBoxForNode( const aiNode *nd, aiVector3D &min, aiVector3D &max, aiMatrix4x4 &trafo );

    // pobranie rozmiarw bryy otaczajcej model (prostopadocianu)
    void GetBoundingBox( aiVector3D &min, aiVector3D &max );

    // sprawdzenie, czy s (p)przezroczyste materiay
    void IsOpacity( const aiNode* nd, unsigned int &currentMesh );

    // sprawdzenie, czy jest aktywna teselacja
    bool IsTesselation();

    // sprawdzenie, czy s dostepne zmienne jednorodne obsugujce parametry materiaw
    void IsUniforms( bool uniforms[4] );

    // struktura opisujca pojedyncz siatk w obiekcie 3D
    // (grup wieloktw o wsplnych atrybutach)
    struct MESH
    {
        // VBO - wsprzdne wierzchoka
        GLuint position;

        // VBO - wsprzdne wektora normalnego
        GLuint normal;

        // VBO - wsprzdne tekstury
        GLuint texCoord;

        // VBO - wsprzdne wektora stycznego
        GLuint tangent;

        // VBO - wsprzdne wektora bistycznego
        GLuint bitangent;

        // EAB - indeksy wierzchokw
        GLuint indices;

        // VAO - tablica wierzchokw
        GLuint vertexArray;

        // liczba indeksw wierzchokw / 3
        unsigned int pointsFaces;

        // konstruktor
        MESH():
            position( 0 ), normal( 0 ), texCoord ( 0 ), tangent ( 0 ), bitangent( 0 ),
            indices( 0 ), vertexArray( 0 ), pointsFaces( 0 ) {}
    };

    // struktura z identyfikatorami obiektw tekstury materiau
    struct MATERIAL_TEXTUREID
    {
        // wspczynnik odbicia wiata otoczenia; oznaczenie Ma
        GLuint ambient;

        // wspczynnik odbicia wiata rozproszonego; oznaczenie Md
        GLuint diffuse;

        // wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
        GLuint specular;

        // wykadnik wspczynnika funkcji rozbysku;
        // zakres wartoci <0;128>; oznaczenie Msh
        GLuint shininess;

        // mapa wysokoci
        GLuint height;

        // mapa wektorw normalnych
        GLuint normal;

        // konstruktor
        MATERIAL_TEXTUREID():
            ambient( 0 ), diffuse( 0 ), specular ( 0 ), shininess( 0 ), height( 0 ), normal () {}
    };

    // obiekt 3D
    const aiScene *assimpScene;

    // czna liczba siatek w pliku
    unsigned int totalMesh;

    // tablica z danymi siatek
    std::vector< MESH > allMeshes;

    // rozmiar modelu
    float size;

    // znacznik (p)przezroczystych materiaw
    bool opacity;

    // wsprzdne rodka modelu
    glm::vec3 center;

    // mapa powiza pomidzy plikami z teksturami
    // i identyfikturami obiektw tekstury
    std::map< std::string, GLuint > textureIdMap;

    // identyfikatory obiektw tekstury
    std::vector< GLuint > textureIds;

    // tablica stuktur z identyfikatorami obiektw tekstury materiau
    std::vector< MATERIAL_TEXTUREID > textureOfEachMaterial;
};

#endif  // __MODELS__H__
