//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// GLSL 1.50
//////////////////////////////////////////////////////////////////////
#version 150

//////////////////////////////////////////////////////////////////////
// struktura opisujca parametry rda wiata
//////////////////////////////////////////////////////////////////////
struct LightSourceParameters
{
    vec4 ambient;               // intensywno wiata otoczenia; oznaczenie La
    vec4 diffuse;               // intensywno wiata rozproszonego; oznaczenie Ld
    vec4 specular;              // intensywno wiata zwierciadlanego; oznaczenie Ls
    vec4 position;              // pooenie rda wiata punktowego (position.w = 1.0)
                                // lub wektor kierunku wiata kierunkowego, skierowany
                                // do rda wiata (position.w = 0.0)
    float constantAttenuation;  // stay wspczynnik tumienia wiata; oznaczenie kc
    float linearAttenuation;    // liniowy wspczynnik tumienia wiata; oznaczenie kl
    float quadraticAttenuation; // kwadratowy wspczynnik tumienia wiata; oznaczenie kq
};

//////////////////////////////////////////////////////////////////////
// prototypy funkcji
//////////////////////////////////////////////////////////////////////
LightSourceParameters LightSource( const int light );
vec4 GlobalAmbientLight( const vec4 ambient );
vec4 LocalAmbientLight( const int light, const vec4 ambient );

//////////////////////////////////////////////////////////////////////
// wiato kierunkowe w modelu Lamberta
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// wektor kierunku wiata lightSource[light].position.xyz musi by
// znormalizowany i wyraony w tym samym ukadzie wsprzdnych co wektor
// normalny i wsprzdne wierzchoka; ponadto wektor ten
// musi by zorientowany w kierunku do rda wiata
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
//////////////////////////////////////////////////////////////////////
vec4 LambertDirectionalLight( const int light, const vec3 normal,
                              const vec3 position, const vec4 diffuse )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightSource.position.xyz );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
        // kocowe obliczenie skadowych owietlenia
        return NdotL * diffuse * lightSource.diffuse;
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// wiato punktowe w modelu Lamberta
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// pooenie rda wiata lightSource[light].position.xyz musi by
// wyraone w tym samym ukadzie wsprzdnych co wektor normalny
// i wsprzdne wierzchoka
// ambient - wspczynnik odbicia wiata otoczenia; oznaczenie Ma
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
//////////////////////////////////////////////////////////////////////
vec4 LambertPointLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // wektor z biecego wierzchoka do rda wiata - wektor kierunku rda wiata
    vec3 lightVec = lightSource.position.xyz - position;

    // obliczenie odlegoci od biecego wierzchoka do rda wiata
    float distance = length( lightVec );

    // normalizacja wektora kierunku rda wiata
    lightVec = normalize( lightVec );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightVec );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczenie wspczynnika tumienia wiata
        float att = 1.0 / ( lightSource.constantAttenuation +
                            lightSource.linearAttenuation * distance +
                            lightSource.quadraticAttenuation * distance * distance );

        // kocowe obliczenie skadowych owietlenia
        return att * ( LocalAmbientLight( light, ambient ) + NdotL * diffuse * lightSource.diffuse );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// owietlenie Lamberta
// wymogi dotyczce parametrw jak w opisach funkcji:
// LambertDirectionalLight i LambertPointLight
//////////////////////////////////////////////////////////////////////
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                    const vec4 ambient, const vec4 diffuse )
{
    // wiato kierunkowe
    if( LightSource( light ).position.w == 0.0 )
        return LambertDirectionalLight( light, normal, position, diffuse ) +
                LocalAmbientLight( light, ambient ) + GlobalAmbientLight( ambient );
    else
    // wiato punktowe
        return LambertPointLight( light, normal, position, ambient, diffuse ) +
                GlobalAmbientLight( ambient );
}

//////////////////////////////////////////////////////////////////////
// wiato kierunkowe w modelu Lamberta z przemieszczeniem
// wektora/kierunku wiata
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// wektor kierunku wiata lightSource[light].position.xyz musi by
// znormalizowany i wyraony w tym samym ukadzie wsprzdnych co wektor
// normalny i wsprzdne wierzchoka; ponadto wektor ten
// musi by zorientowany w kierunku do rda wiata
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 LambertDirectionalLight( const int light, const vec3 normal,
                              const vec3 position, const vec4 diffuse,
                              const mat3 TBN )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, TBN * lightSource.position.xyz );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
        // kocowe obliczenie skadowych owietlenia
        return NdotL * diffuse * lightSource.diffuse;
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// wiato punktowe w modelu Lamberta z przemieszczeniem
// wektora/kierunku wiata
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// pooenie rda wiata lightSource[light].position.xyz musi by
// wyraone w tym samym ukadzie wsprzdnych co wektor normalny
// i wsprzdne wierzchoka
// ambient - wspczynnik odbicia wiata otoczenia; oznaczenie Ma
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 LambertPointLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse, const mat3 TBN )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // wektor z biecego wierzchoka do rda wiata - wektor kierunku rda wiata
    vec3 lightVec = lightSource.position.xyz - position;

    // obliczenie odlegoci od biecego wierzchoka do rda wiata
    float distance = length( lightVec );

    // normalizacja wektora kierunku rda wiata
    lightVec = TBN * normalize( lightVec );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightVec );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczenie wspczynnika tumienia wiata
        float att = 1.0 / ( lightSource.constantAttenuation +
                            lightSource.linearAttenuation * distance +
                            lightSource.quadraticAttenuation * distance * distance );

        // kocowe obliczenie skadowych owietlenia
        return att * ( LocalAmbientLight( light, ambient ) + NdotL * diffuse * lightSource.diffuse );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// owietlenie Lamberta z przemieszczeniem wektora/kierunku wiata
// wymogi dotyczce parametrw jak w opisach funkcji:
// LambertDirectionalLight i LambertPointLight
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                    const vec4 ambient, const vec4 diffuse, const mat3 TBN )
{
    // wiato kierunkowe
    if( LightSource( light ).position.w == 0.0 )
        return LambertDirectionalLight( light, normal, position, diffuse, TBN ) +
                LocalAmbientLight( light, ambient ) + GlobalAmbientLight( ambient );
    else
    // wiato punktowe
        return LambertPointLight( light, normal, position, ambient, diffuse, TBN ) +
                GlobalAmbientLight( ambient );
}
