//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// GLSL 1.50
//////////////////////////////////////////////////////////////////////
#version 150

//////////////////////////////////////////////////////////////////////
// struktura opisujca parametry materiau
//////////////////////////////////////////////////////////////////////
struct MaterialParameters
{
    vec4 ambient;       // wspczynnik odbicia wiata otoczenia; oznaczenie Ma
    vec4 diffuse;       // wspczynnik odbicia wiata rozproszonego; oznaczenie Md
    vec4 specular;      // wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
    float shininess;    // wykadnik wspczynnika funkcji rozbysku;
                        // zakres wartoci <0;128>; oznaczenie Msh
};

//////////////////////////////////////////////////////////////////////
// prototypy funkcji
//////////////////////////////////////////////////////////////////////
MaterialParameters GetMaterial( const int material );
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse,
                        const vec4 specular, const float shininess );
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                    const vec4 ambient, const vec4 diffuse );

//////////////////////////////////////////////////////////////////////
// rozmiar tablicy z mask filtra 5x5
//////////////////////////////////////////////////////////////////////
#define FILTER_5X5_SIZE 25

//////////////////////////////////////////////////////////////////////
// generator liczb pseudolosowych
// seed - zarodek generatora
// http://www.geeks3d.com/20100831/shader-library-noise-and-pseudo-random-number-generator-in-glsl/
//////////////////////////////////////////////////////////////////////
float rand( const vec2 seed )
{
    int n = int( seed.x * 40.0 + seed.y * 6400.0 );
    n = (n << 13) ^ n;
    return 1.0 - float( (n * (n * n * 15731 + 789221) + 1376312589) & 0x7fffffff) / 1073741824.0;
}

//////////////////////////////////////////////////////////////////////
// numer materiau
//////////////////////////////////////////////////////////////////////
uniform int material;

//////////////////////////////////////////////////////////////////////
// tekstura gbokoci z map cienia
//////////////////////////////////////////////////////////////////////
uniform sampler2DShadow shadowTex;

//////////////////////////////////////////////////////////////////////
// interpolowany wektor normalny
//////////////////////////////////////////////////////////////////////
in vec3 inoutNormal;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wierzchoka
//////////////////////////////////////////////////////////////////////
in vec3 inoutPosition;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
in vec4 inoutTexCoord;

//////////////////////////////////////////////////////////////////////
// wyjciowy kolor fragmentu
//////////////////////////////////////////////////////////////////////
out vec4 outColor;

void main()
{
    // normalizacja wektora normalnego zalena od strony wielokta
    vec3 normal = gl_FrontFacing ? normalize( inoutNormal ) : -normalize( inoutNormal );

    // pobranie wybranego materiau
    MaterialParameters mat = GetMaterial( material );

    // kolor owietlonego fragmentu
    vec4 lightColor = BlinnPhongLight( 0, normal, inoutPosition, mat.ambient, mat.diffuse, mat.specular, mat.shininess );

    // kolor fragmentu w cieniu
    vec4 shadowColor = LambertLight( 0, normal, inoutPosition, mat.ambient, mat.diffuse );

    // rozmiar tekstury z map cienia
    vec2 shadowTexSize = textureSize( shadowTex, 0 );

    // prbkowanie tekstury cienia
    float shadow = 0.0;
    for( int i = 0; i < FILTER_5X5_SIZE; i++ )
    {
        // wsprzdne z przedziau <-2.5 ; 2,5>
        vec2 offset = 5.0f * vec2( rand( gl_FragCoord.xy ), rand( gl_FragCoord.yx ) ) - 2.5f;

        // zawijanie wsprzdnych do okrgu o promieniu 2,5
        float len = length( offset );
        if( len > 2.5f )
            offset *= (2.5f / len);

        // przeksztacenie wsprzdnych do przestrzeni wsprzdnych tekstury
        offset /= shadowTexSize;
        shadow += textureProj( shadowTex, inoutTexCoord + vec4( offset, 0, 0 ) );
    }
    shadow /= FILTER_5X5_SIZE;

    // kolor fragmentu z cieniem pomnoonym przez kolor cienia
    outColor = mix( 1.5 * shadowColor * vec4( 0.184314f, 0.309804f, 0.309804f, 1.000000f ), lightColor, shadow );
}
