//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "sphere.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 6.0f;
GLfloat far = 14.0f;

//////////////////////////////////////////////////////////////////////
// wspczynniki rwna poszczeglnych paszczyzn zawierajcych
// ciany; wsprzdne w przestrzeni obserwatora
//////////////////////////////////////////////////////////////////////
glm::vec4 wallPlanes[5] =
{
    // podoga
    glm::vec4( 0.0f, 1.0f, 0.0f, -bottom ),

    // lewa ciana
    glm::vec4( 1.0f, 0.0f, 0.0f, -left ),

    // prawa ciana
    glm::vec4( -1.0f, 0.0f, 0.0f, right ),

    // sufit
    glm::vec4( 0.0f, -1.0f, 0.0f, top ),

    // tylna ciana
    glm::vec4( 0.0f, 0.0f, 1.0f, far - 1.0f )
};

//////////////////////////////////////////////////////////////////////
// wektory normalne wierzchokw trjktw tworzcych ciany pokoju
//////////////////////////////////////////////////////////////////////
GLfloat roomNormal[5*6*3] =
{
    // podoga
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,

    // lewa ciana
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,

    // prawa ciana
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,

    // sufit
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,

    // tylna ciana
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw tworzcych ciany pokoju;
// wsprzdne w przestrzeni obserwatora
//////////////////////////////////////////////////////////////////////
GLfloat roomPosition[5*6*4] =
{
    // podoga
    left, bottom, -near, 1.0f,
    right, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,

    // lewa ciana
    left, bottom, -near, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,

    // prawa ciana
    right, bottom, -near, 1.0f,
    right, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,

    // sufit
    left, top, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, top, -near, 1.0f,

    // tylna ciana
    left, bottom, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.75f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// pooenie rda wiata punktowego we wsprzdnych obserwatora
//////////////////////////////////////////////////////////////////////
glm::vec4 lightPosition( 3.0f, 3.0f, 3.0f - (near+far)/2.0f, 1.0f );

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    OBJECT,         // czajnik i ciany
    LIGHT_POINT,    // pooenie rda wiata
    SHADOW,         // cie
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_POSITION,
    TEAPOT_NORMAL,
    LIGHT_POINT_POSITION,
    ROOM_POSITION,
    ROOM_NORMAL,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    // LIGHT_POINT,
    ROOM = 2,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_INDICES,
    LIGHT_POINT_INDICES,
    INDICES_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0
#define NORMAL   1

//////////////////////////////////////////////////////////////////////
// funkcja obliczajca macierz cienia - macierz rzutowania obiektu
// na paszczyzn, na ktrej pojawia si cie; jest to zwyka macierz
// rzutowania perspektywicznego, z t rnic, e pooeniem obserwatora
// jest w tym wypadku pooenie rda wiata; przemnoenie tej macierzy
// przez macierz modelowania spowoduje "spaszczenie" obiektw na paszczynie,
// czyli uzyskanie efektu cienia
// wallPlane - wspczynniki rwnania paszczyzny, na ktrej bdzie cie
//////////////////////////////////////////////////////////////////////
glm::mat4 ShadowMatrix( const glm::vec4 &wallPlane )
{
    // iloczyn skalarny wektora zoonego ze wspczynnikw
    // rwnania paszczyzny i pooenia rda wiata
    float dot = glm::dot( wallPlane, lightPosition );

    // obliczenie macierzy rzutowania
    return glm::mat4
        (
            // pierwsza kolumna
            dot  - lightPosition[0] * wallPlane[0],
            0.0f - lightPosition[1] * wallPlane[0],
            0.0f - lightPosition[2] * wallPlane[0],
            0.0f - lightPosition[3] * wallPlane[0],

            // druga kolumna
            0.0f - lightPosition[0] * wallPlane[1],
            dot  - lightPosition[1] * wallPlane[1],
            0.0f - lightPosition[2] * wallPlane[1],
            0.0f - lightPosition[3] * wallPlane[1],

            // trzecia kolumna
            0.0f - lightPosition[0] * wallPlane[2],
            0.0f - lightPosition[1] * wallPlane[2],
            dot  - lightPosition[2] * wallPlane[2],
            0.0f - lightPosition[3] * wallPlane[2],

            // czwarta kolumna
            0.0f - lightPosition[0] * wallPlane[3],
            0.0f - lightPosition[1] * wallPlane[3],
            0.0f - lightPosition[2] * wallPlane[3],
            dot  - lightPosition[3] * wallPlane[3]
        );
}

//////////////////////////////////////////////////////////////////////
// rysowanie czajnika
// modelViewMatrix - macierz modelu
//////////////////////////////////////////////////////////////////////
void DrawTeapot( glm::mat4x4 &modelViewMatrix )
{
    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // przeksztacenie pooenia rda wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[OBJECT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[OBJECT], "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program[OBJECT], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[OBJECT], "material" ), material );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie programu
    glUseProgram( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie cian pokoju
//////////////////////////////////////////////////////////////////////
void DrawRoom()
{
    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // wczenie programu
    glUseProgram( program[OBJECT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    // z uwagi na to, e wsprzdne wierzchokw trjktw tworzcych ciany
    // pokoju s ju w przestrzeni obserwatora, nie ma potrzeby mnoenia macierzy
    // rzutowania przez macierz modelu-widoku
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );

    // pooenie obserwatora w ukadzie wsprzdnych obserwatora
    GLfloat eyePosition[4] = { 0.0f, 0.0f, 0.0f, 1.0f };

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obserwatora
    glUniform4fv( glGetUniformLocation( program[OBJECT], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[OBJECT], "eyePosition" ), 1, eyePosition );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[OBJECT], "material" ), MTL_DEFAULT );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLES, 0, 30 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie programu
    glUseProgram( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie cienia
// modelViewMatrix - macierz modelu
//////////////////////////////////////////////////////////////////////
void DrawShadow( glm::mat4x4 &modelViewMatrix )
{
    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[SHADOW] );

    // nazwy poszczeglnych macierzy w shaderze geometrii
    const char *matrixName[5] =
    {
        "wallPlanesModelViewProjectionMatrix[0]",
        "wallPlanesModelViewProjectionMatrix[1]",
        "wallPlanesModelViewProjectionMatrix[2]",
        "wallPlanesModelViewProjectionMatrix[3]",
        "wallPlanesModelViewProjectionMatrix[4]"
    };

    // nazwy poszczeglnych paszczyzn obcinania w shaderze geometrii
    const char *clipPlaneName[5] =
    {
        "clipPlane[0]",
        "clipPlane[1]",
        "clipPlane[2]",
        "clipPlane[3]",
        "clipPlane[4]"
    };

    // zaadowanie zmiennych jednorodnych - iloczynw macierzy modelu-widoku,
    // macierzy cienia i rzutowania oraz paszczyzn obcinania
    for( int i = 0; i < 5; i++ )
    {
        glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * ShadowMatrix( wallPlanes[i] ) * modelViewMatrix;
        glUniformMatrix4fv( glGetUniformLocation( program[SHADOW], matrixName[i] ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );
        glUniform4fv( glGetUniformLocation( program[SHADOW], clipPlaneName[i] ), 1, glm::value_ptr( wallPlanes[i] ) );
    }

    // zaadowanie zmiennej jednorodnej - macierzy modelu-widoku obiektu
    glUniformMatrix4fv( glGetUniformLocation( program[SHADOW], "modelViewMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewMatrix ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie rda wiata
//////////////////////////////////////////////////////////////////////
void DrawLight()
{
    // transformacje pooenia rda wiata punktowego
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( lightPosition ) );
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( 0.2f, 0.2f, 0.2f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // wczenie programu
    glUseProgram( program[LIGHT_POINT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_POINT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, SPHERE_LOW_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // rysowanie cian pokoju
    DrawRoom();

    // rysowanie cienia
    DrawShadow( modelViewMatrix );

    // rysowanie czajnika
    DrawTeapot( modelViewMatrix );

    // rysowanie rda wiata
    DrawLight();

    // wyczenie testu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wypisanie pooenia rda wiata punktowego
    const glm::vec4 clGreen( 0.000000f, 0.501961f, 0.000000f, 1.000000f );
    std::ostringstream txt;
    txt << std::setprecision( 1 ) << std::fixed
        << "pooenie rda wiata: (" << lightPosition[0] << ";"
        << lightPosition[1] << ";" << lightPosition[2] + (near+far)/2.0f << ")";
    DrawText8x16( 3, 3, txt.str(), clGreen );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "materia: " ) + GetMaterialName( material ), clGreen );

    // wczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT] = glCreateProgram();
    glAttachShader( program[OBJECT], LoadShader( GL_VERTEX_SHADER, "swiatlo_fragment_vs.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_fragment_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_POINT] = glCreateProgram();
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_VERTEX_SHADER, "punkt_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_FRAGMENT_SHADER, "punkt_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_POINT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW] = glCreateProgram();
    glAttachShader( program[SHADOW], LoadShader( GL_VERTEX_SHADER, "cien_plaski_vs.glsl" ) );
    glAttachShader( program[SHADOW], LoadShader( GL_GEOMETRY_SHADER, "cien_plaski_gs.glsl" ) );
    glAttachShader( program[SHADOW], LoadShader( GL_FRAGMENT_SHADER, "cien_plaski_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW] );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[LIGHT_POINT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[LIGHT_POINT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[LIGHT_POINT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereLowPositionNormal ), sphereLowPositionNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[LIGHT_POINT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[LIGHT_POINT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereLowIndices ), sphereLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[ROOM] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomPosition ), roomPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomNormal ), roomNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie paszczyzn obcinania
    glEnable( GL_CLIP_DISTANCE0 );
    glEnable( GL_CLIP_DISTANCE1 );
    glEnable( GL_CLIP_DISTANCE2 );
    glEnable( GL_CLIP_DISTANCE3 );
    glEnable( GL_CLIP_DISTANCE4 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[OBJECT] );
    glDeleteProgram( program[LIGHT_POINT] );
    glDeleteProgram( program[SHADOW] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
