//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "textures.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// nazwa pliku podana w wierszu polecenia
//////////////////////////////////////////////////////////////////////
char *fileName = NULL;

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 70.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 15.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na obiekt
//////////////////////////////////////////////////////////////////////
GLfloat floorPosition[6*3] =
{
    -50.0f, -1.0f, -50.0f,
    50.0f, -1.0f, -50.0f,
    50.0f, -1.0f, 50.0f,
    -50.0f, -1.0f, -50.0f,
    50.0f, -1.0f, 50.0f,
    -50.0f, -1.0f, 50.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
GLfloat floorTexCoord[6*2] =
{
    0.0f, 0.0f,
    10.0f, 0.0f,
    10.0f, 10.0f,
    0.0f, 0.0f,
    10.0f, 10.0f,
    0.0f, 10.0f
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture;

//////////////////////////////////////////////////////////////////////
// filtr pomniejszajcy
//////////////////////////////////////////////////////////////////////
GLuint minFilter = GL_NEAREST;

//////////////////////////////////////////////////////////////////////
// maksymalny stopie anizotropii
//////////////////////////////////////////////////////////////////////
GLfloat maxAnisotropy = -1.0f;

//////////////////////////////////////////////////////////////////////
// znacznik wczenia filtracji anizotropowej
//////////////////////////////////////////////////////////////////////
bool anisotropicFilter = false;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wybr tekstury 2D
    glBindTexture( GL_TEXTURE_2D, texture );

    // filtracja anizotropowa
    if( maxAnisotropy > 0.0f )
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, anisotropicFilter == true ? maxAnisotropy : 1.0f );
    else
    {
        // wyczenie filtracji anizotropowej
        if( maxAnisotropy > 0.0f )
            glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 1.0f );

        // filtr pomniejszajcy
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, minFilter );
    }

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program ,"tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLES, 0, 6 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie tekstury 2D
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wywietlenie informacji o trybie filtracji tekstury
    glDisable( GL_DEPTH_TEST );
    const glm::vec4 yellowGreen( 0.603922f, 0.803922f, 0.196078f, 1.000000f );
    if( anisotropicFilter == true && maxAnisotropy > 0.0f )
        DrawText8x16( 3, 3, "filtracja anizotropowa", yellowGreen );
    else
        switch( minFilter )
        {
            case GL_NEAREST: DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_NEAREST", yellowGreen ); break;
            case GL_LINEAR:  DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_LINEAR", yellowGreen ); break;
            case GL_NEAREST_MIPMAP_NEAREST: DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_NEAREST_MIPMAP_NEAREST", yellowGreen ); break;
            case GL_NEAREST_MIPMAP_LINEAR: DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_NEAREST_MIPMAP_LINEAR", yellowGreen ); break;
            case GL_LINEAR_MIPMAP_NEAREST: DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_LINEAR_MIPMAP_NEAREST", yellowGreen ); break;
            case GL_LINEAR_MIPMAP_LINEAR: DrawText8x16( 3, 3, "filtr pomniejszajcy: GL_LINEAR_MIPMAP_LINEAR", yellowGreen ); break;
        }
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // pobranie maksymalnego stopnia anizotropii
    glGetFloatv( GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &maxAnisotropy );

    // sprawdzenie, czy nie wystpi bd, jeeli tak, to nie ma
    // dostpnego rozszerzenia GL_EXT_texture_filter_anisotropic
    if( glGetError() != GL_NO_ERROR )
        maxAnisotropy = -1.0;

    // kolor ta - zawarto bufora koloru
    glClearColor( 0.9f, 0.9f, 1.0f, 1.0f );

    // utworzenie obiektu tekstury
    glGenTextures( 1, &texture );
    glBindTexture( GL_TEXTURE_2D, texture );

    // wczytanie tekstury
    if( !LoadTexture( fileName, GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileName << std::endl;
        exit( 0 );
    }

    // pobranie wewntrznego formatu danych tekstury
    GLint internalFormat;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &internalFormat );
    if( internalFormat == GL_RED )
    {
        // podstawianie skadowych tekseli: RGBA -> RRRA
        GLint swizzle[4] = { GL_RED, GL_RED, GL_RED, GL_ALPHA };
        glTexParameteriv( GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzle );
    }

    // generowanie mipmap
    glGenerateMipmap( GL_TEXTURE_2D );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "filtracja_tekstury2d_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "filtracja_tekstury2d_fs.glsl" ) );
    LinkValidateProgram( program );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( floorPosition ), floorPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( floorTexCoord ), floorTexCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( 1, &texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
