//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <cstring>
#include <sstream>
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <iomanip>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"
#include "models.h"

//////////////////////////////////////////////////////////////////////
// nazwa pliku podana w wierszu polecenia
//////////////////////////////////////////////////////////////////////
char *fileName = NULL;

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// rodzaj modelu owietlenia Phonga/Blinna-Phonga/Lamberta
//////////////////////////////////////////////////////////////////////
enum
{
    PHONG_LIGHT,
    BLINN_PHONG_LIGHT,
    LAMBERT_LIGHT,
    PHONG_LIGHT_NORMAL_MAP,
    BLINN_PHONG_LIGHT_NORMAL_MAP,
    LAMBERT_LIGHT_NORMAL_MAP
};
int lightType = BLINN_PHONG_LIGHT_NORMAL_MAP;

//////////////////////////////////////////////////////////////////////
// znacznik odwrcenia wartoci mapy wysokoci
//////////////////////////////////////////////////////////////////////
bool reverseHeightMap = true;

//////////////////////////////////////////////////////////////////////
// obiekt klasy obsugujcej odczyt i rendering plikw 3D
//////////////////////////////////////////////////////////////////////
ModelAndTextureLoader model;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, -(near+far)/2.0f ) );

    // skalowanie obiektu do wielkoci bryy obcinania
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( (right - left) * 0.8f / model.GetSize() ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wycentowanie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, -model.GetCenter() );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // pobranie indeksu funkcji obsugujcej wybrany rodzaj modelu owietlenia
    const char *lightName[] = { "Phong", "BlinnPhong", "Lambert", "PhongNormalMap", "BlinnPhongNormalMap", "LambertNormalMap" };
    GLuint light = glGetSubroutineIndex( program, GL_FRAGMENT_SHADER, lightName[lightType] );

    // zaadowanie zmiennej jednorodnej podprogramu - rodzaj modelu owietlenia
    glUniformSubroutinesuiv( GL_FRAGMENT_SHADER, 1, &light );

    // wspczynnik skalowania wielkoci przemieszczenia
    GLfloat displacement = 0.015f * model.GetSize();
    glUniform1f( glGetUniformLocation( program, "displacement" ), displacement );

    // znacznik odwrcenia wartoci mapy wysokoci
    glUniform1i( glGetUniformLocation( program, "reverseHeightMap" ), reverseHeightMap );

    // obiekt zawiera materiay pprzezroczyste
    if( model.GetOpacity() )
    {
        // narysowanie danych obiektw nieprzezroczystych
        model.MeshRenderingOpacity( false );

        // wczenie mieszania kolorw
        glEnable( GL_BLEND );

        // wyczenie zapisu danych bufora gbokoci
        glDepthMask( GL_FALSE );

        // wczenie renderingu wybranej strony wielokta
        glEnable( GL_CULL_FACE );

        // rysowanie tylko tylnej strony wieloktw
        glCullFace( GL_FRONT );

        // narysowanie danych (p)przezroczystych
        model.MeshRenderingOpacity( true );

        // rysowanie tylko przedniej strony wieloktw
        glCullFace( GL_BACK );

        // narysowanie danych (p)przezroczystych
        model.MeshRenderingOpacity( true );

        // wyczenie renderingu wybranej strony wielokta
        glDisable( GL_CULL_FACE );

        // wczenie zapisu danych do bufora gbokoci
        glDepthMask( GL_TRUE );

        // wyczenie mieszania kolorw
        glDisable( GL_BLEND );
    }
    else
        // rendering obiektu 3D z pliku
        model.MeshRendering();

    // wyczenie programu
    glUseProgram( 0 );

    // wypisanie wektora kierunku rda wiata
    std::ostringstream txt;
    txt << std::setprecision( 4 ) << std::fixed
        << "Wektor kierunku wiata: (" << lightPosition[0] << ";" << lightPosition[1] << ";" << lightPosition[2] << ")";
    DrawText8x16( 3, 3, txt.str() );

    // wypisanie rodzaju modelu owietlenia
    const char* ligtTypeStr[] = { "owietlenie Phonga",
                                  "owietlenie Blinna-Phonga",
                                  "owietlenie Lamberta",
                                  "owietlenie Phonga z mapowaniem wektorw normalnych",
                                  "owietlenie Blinna-Phonga z mapowaniem wektorw normalnych",
                                  "owietlenie Lamberta z mapowaniem wektorw normalnych" };
    DrawText8x16( 3, 21, ligtTypeStr[lightType] );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // zaadowanie danych z pliku 3D
    if( !model.Load( fileName ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileName << std::endl;
        exit( 0 );
    }

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "mapowanie_przemieszczen_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_TESS_CONTROL_SHADER, "mapowanie_przemieszczen_tcs.glsl" ) );
    glAttachShader( program, LoadShader( GL_TESS_EVALUATION_SHADER, "mapowanie_przemieszczen_tes.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "mapowanie_przemieszczen_fs.glsl" ) );
    LinkValidateProgram( program );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();

    // wczenie ustawie uywanych podczas renderingu obiektu pprzezroczystego
    if( model.GetOpacity() )
    {
        // wspczynniki mieszania kolorw
        glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    }
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
